package models

import (
	"database/sql"
	"encoding/json"
	"fmt"
	"net/url"
	"sort"
	"strconv"
	"strings"
	"time"
	"unicode/utf8"

	"code.justin.tv/chat/db"
	"code.justin.tv/common/golibs/errorlogger"
	"code.justin.tv/vod/vinyl/datastore/vinyldb/utils"
	"code.justin.tv/vod/vinyl/errors"
	"code.justin.tv/vod/vinyl/markdown"
	"code.justin.tv/vod/vinyl/models"
	"github.com/lib/pq"
	"github.com/lib/pq/hstore"
)

// Constants associated with the vod table and its fields.
var (
	VodCreateFields = []string{
		"bitrates", "broadcast_id", "broadcast_type", "broadcaster_software",
		"codecs", "created_at", "created_by_id", "deleted", "delete_at",
		"description", "description_html", "display_names", "duration", "formats", "fps",
		"game", "language", "manifest",
		"\"offset\"", "owner_id", "origin",
		"playlist_preferences", "published_at", "resolutions", "source_archive_id",
		"started_on", "status", "tag_hstore",
		"title", "updated_at", "uri", "views_count", "viewable_at", "viewable", "selected_thumbnail_id",
	}
	VodFields           = append([]string{"id"}, VodCreateFields...)
	VodTableName        = "vods"
	TranscodingStatuses = []string{models.StatusUploading, models.StatusPendingTranscode, models.StatusTranscoding}
	Statuses            = append([]string{models.StatusCreated, models.StatusRecording, models.StatusRecorded, models.StatusUnprocessed, models.StatusFailed}, TranscodingStatuses...)
	MaxCharsInAllTags   = 500
	MaxCharsPerTag      = 100
)

// FetchAllVODFieldsQuery returns a string used to query for all columns in the vod table.
func FetchAllVODFieldsQuery() string {
	return "SELECT " + strings.Join(VodFields, ",") + " FROM " + VodTableName
}

// DeleteVodsQuery returns a string used to update vod entries to be deleted.
func DeleteVodsQuery() string {
	return "UPDATE " + VodTableName + " SET deleted = true"
}

// DestroyVodsQuery returns a string used to remove vod entries from the table.
func DestroyVodsQuery() string {
	return "DELETE FROM " + VodTableName
}

// AppendVodFilters adds filters to a query
func AppendVodFilters(query []interface{}, filters *models.VodFilters) []interface{} {
	if filters == nil {
		return query
	}
	if !filters.IncludeDeleted {
		query = append(query, Active())
	}
	if !filters.IncludePrivate {
		query = append(query, Published())
	}
	if !filters.IncludeProcessing {
		query = append(query, Processed())
	}
	return query
}

// Active returns a constraint that vods must not be marked for deletion.
func Active() string {
	return "AND deleted IS NOT TRUE"
}

// Published returns a filter for vods that are published.
func Published() string {
	return "AND viewable IS DISTINCT FROM 'private'"
}

// Processed returns a filter for vods that have completed processing
func Processed() string {
	return "AND status IN ('recording', 'recorded')"
}

// Watchable returns a filter for vods that are active and published.
func Watchable() string {
	return Active() + " " + Published() + " " + Processed()
}

// Vod is the representation of a vod in the VinylDB database.
// TODO: Determine omitempty
type Vod struct {
	BroadcastID         int
	BroadcastType       string
	BroadcasterSoftware sql.NullString
	CreatedAt           time.Time
	CreatedBy           sql.NullInt64
	Deleted             sql.NullBool
	DeleteAt            pq.NullTime
	Description         sql.NullString
	DescriptionHTML     sql.NullString
	Duration            int
	Fps                 sql.NullString
	Game                sql.NullString
	ID                  int64
	Language            sql.NullString
	Manifest            sql.NullString
	Offset              int
	OwnerID             int64
	Origin              sql.NullString
	Resolutions         sql.NullString
	SelectedThumbnailID sql.NullInt64
	SourceArchiveID     sql.NullInt64
	StartedOn           time.Time
	Status              string
	TagHstore           hstore.Hstore
	Thumbnails          VodThumbnails
	Title               sql.NullString
	TotalLength         int
	UpdatedAt           time.Time
	URI                 string
	Viewable            sql.NullString
	ViewableAt          pq.NullTime
	Views               int64
	PublishedAt         pq.NullTime

	// Unexported fields
	bitrates            sql.NullString
	codecs              sql.NullString
	displayNames        sql.NullString
	formats             sql.NullString
	playlistPreferences sql.NullString

	// TODO: Deprecate this
	APIID string
}

const (
	thumbnail404  = "https://vod-secure.twitch.tv/_404/404_processing_%{width}x%{height}.png"
	s3Prefix      = "s3_vods/"
	thumbnailHost = "https://static-cdn.jtvnw.net/"
	size          = "%{width}x%{height}"

	countessURL = "https://countess.twitch.tv/ping.gif"
)

// FromVinylVod converts a VinylVod object into a VinylDB vod object used to interact with the db.
func FromVinylVod(vod *models.Vod) (*Vod, error) {
	var err error
	ret := &Vod{
		BroadcastID:         vod.BroadcastID,
		BroadcastType:       vod.BroadcastType,
		BroadcasterSoftware: sql.NullString(vod.BroadcasterSoftware),
		CreatedAt:           vod.CreatedAt,
		CreatedBy:           sql.NullInt64(vod.CreatedBy),
		Deleted:             sql.NullBool(vod.Deleted),
		DeleteAt:            vod.DeleteAt.AsPQ(),
		Description:         sql.NullString(vod.Description),
		Duration:            vod.Duration,
		Game:                sql.NullString(vod.Game),
		Language:            sql.NullString(vod.Language),
		Manifest:            sql.NullString(vod.Manifest),
		Offset:              vod.Offset,
		OwnerID:             vod.OwnerID,
		Origin:              sql.NullString(vod.Origin),
		SourceArchiveID:     sql.NullInt64(vod.SourceArchiveID),
		Status:              vod.Status,
		Title:               sql.NullString(vod.Title),
		UpdatedAt:           vod.UpdatedAt,
		URI:                 vod.URI,
		Views:               vod.Views,

		StartedOn:   vod.StartedOn,
		TotalLength: vod.TotalLength,
		ViewableAt:  vod.ViewableAt.AsPQ(),
		Viewable:    sql.NullString(vod.Viewable),
		PublishedAt: vod.PublishedAt.AsPQ(),
	}

	// Parse markdown description into HTML
	if vod.Description.Valid {
		ret.DescriptionHTML = sql.NullString{Valid: true, String: markdown.ConvertMarkdown(vod.Description.String)}
	}

	// Convert `formats` into the unexported fields
	if vod.ShowFormats != nil {
		err = ApplyShowFormats(vod, ret)
		if err != nil {
			return nil, err
		}
	}

	// Convert taglist to taghstore
	if vod.TagList != "" {
		ret.TagHstore, err = TagListToHStore(vod.TagList)
		if err != nil {
			return nil, err
		}
	}

	// Default viewable to 'public', so we don't start hiding new vods by accident
	if !ret.Viewable.Valid || ret.Viewable.String == "" {
		ret.Viewable = sql.NullString{Valid: true, String: "public"}
	}

	return ret, nil
}

// ThumbnailModels returns the vod's structured thumbnails
func (V *Vod) ThumbnailModels() models.Thumbnails {
	thumbnails := models.Thumbnails{}
	for _, thumb := range V.Thumbnails {
		thumbnails = append(thumbnails, thumb.ToVinylVodThumbnail())
	}
	return thumbnails
}

// ApplyShowFormats updates ret properties based on corresponding vod properties
func ApplyShowFormats(vod *models.Vod, ret *Vod) error {
	var err error
	formats := []string{}
	bitrateMap := map[string]int{}
	codecMap := map[string]string{}
	displayNameMap := map[string]string{}
	playlistPreferenceMap := map[string]int{}
	resolutionMap := map[string]string{}
	fpsMap := map[string]float64{}

	for format, showMap := range vod.ShowFormats {
		formats = append(formats, format)
		for key, val := range showMap {
			switch key {
			case "playlist_preference":
				switch v := val.(type) {
				case int64:
					playlistPreferenceMap[format] = int(v)
				case int:
					playlistPreferenceMap[format] = v
				case float64:
					pVal := val.(float64)
					playlistPreferenceMap[format] = int(pVal)
				}
			case "display_name":
				if d, ok := val.(string); ok {
					displayNameMap[format] = d
				} else {
					displayNameMap[format] = ""
				}
			case "bitrate":
				switch v := val.(type) {
				case int64:
					bitrateMap[format] = int(v)
				case int:
					bitrateMap[format] = v
				case float64:
					bVal := val.(float64)
					bitrateMap[format] = int(bVal)
				}
			case "fps":
				switch v := val.(type) {
				case string:
					fVal, err := strconv.ParseFloat(v, 64)
					if err != nil {
						return err
					}
					fpsMap[format] = fVal
				case float64:
					fpsMap[format] = v
				}
			case "codecs":
				if c, ok := val.(string); ok {
					codecMap[format] = c
				} else {
					codecMap[format] = ""
				}
			case "resolution":
				if r, ok := val.(string); ok {
					resolutionMap[format] = r
				} else {
					resolutionMap[format] = ""
				}
			}
		}
	}

	sort.Strings(formats)
	formatStr, err := json.Marshal(formats)
	if err != nil {
		return err
	}
	ret.formats = sql.NullString{Valid: true, String: string(formatStr)}

	bitrateStr, err := json.Marshal(bitrateMap)
	if err != nil {
		return err
	}
	ret.bitrates = sql.NullString{Valid: true, String: string(bitrateStr)}

	codecStr, err := json.Marshal(codecMap)
	if err != nil {
		return err
	}
	ret.codecs = sql.NullString{Valid: true, String: string(codecStr)}

	displayNameStr, err := json.Marshal(displayNameMap)
	if err != nil {
		return err
	}
	ret.displayNames = sql.NullString{Valid: true, String: string(displayNameStr)}

	playlistPrefStr, err := json.Marshal(playlistPreferenceMap)
	if err != nil {
		return err
	}
	ret.playlistPreferences = sql.NullString{Valid: true, String: string(playlistPrefStr)}

	resolutionStr, err := json.Marshal(resolutionMap)
	if err != nil {
		return err
	}
	ret.Resolutions = sql.NullString{Valid: true, String: string(resolutionStr)}

	fpsStr, err := json.Marshal(fpsMap)
	if err != nil {
		return err
	}
	ret.Fps = sql.NullString{Valid: true, String: string(fpsStr)}

	return nil
}

// TagListToHStore converts a comma-separated list of tags into an hstore to be written to the db.
func TagListToHStore(tagList string) (hstore.Hstore, error) {
	if utf8.RuneCountInString(tagList) > MaxCharsInAllTags {
		return hstore.Hstore{}, errors.TagFieldTooLongError{}
	}
	tags := strings.Split(tagList, ",")
	tagsMap := map[string]sql.NullString{}
	for _, t := range tags {
		if utf8.RuneCountInString(t) > MaxCharsPerTag {
			return hstore.Hstore{}, errors.PerTagTooLongError{}
		}
		tagsMap[strings.TrimSpace(t)] = sql.NullString{Valid: false}
	}

	return hstore.Hstore{Map: tagsMap}, nil
}

// AsVinylVod converts a VinylDB vod object into a Vinyl vod object that can be returned.
func (V *Vod) AsVinylVod(options ...AsVinylVodOpt) (*models.Vod, error) {
	// get_vods_by_id is the only endpoint that expects the showFormats field, so don't return the field by default.
	params := asVinylVodParams{
		containShowFormats: false,
	}

	for _, option := range options {
		option(&params)
	}

	var err error

	tags := make([]string, len(V.TagHstore.Map))
	if V.TagHstore.Map != nil {
		i := 0
		for k := range V.TagHstore.Map {
			tags[i] = k
			i++
		}
	}
	descriptionHTML := models.NullString(V.DescriptionHTML)

	sort.StringSlice(tags).Sort()
	res := &models.Vod{
		BroadcastID:         V.BroadcastID,
		BroadcastType:       V.BroadcastType,
		BroadcasterSoftware: models.NullString(V.BroadcasterSoftware),
		CreatedAt:           V.CreatedAt.Round(time.Second),
		CreatedBy:           models.NullInt64(V.CreatedBy),
		DeleteAt:            models.PQAsNullTime(V.DeleteAt),
		Deleted:             models.NullBool(V.Deleted),
		Description:         models.NullString(V.Description),
		DescriptionHTML:     descriptionHTML,
		Duration:            V.Duration,
		Game:                models.NullString(V.Game),
		ID:                  V.ID,
		Language:            models.NullString(V.Language),
		Manifest:            models.NullString(V.Manifest),
		Offset:              V.Offset,
		Origin:              models.NullString(V.Origin),
		OwnerID:             V.OwnerID,
		PublishedAt:         models.PQAsNullTime(V.PublishedAt),
		SourceArchiveID:     models.NullInt64(V.SourceArchiveID),
		StartedOn:           V.StartedOn,
		Status:              V.status(),
		TagList:             strings.Join(tags, ", "),
		Title:               models.NullString(V.Title),
		Thumbnails:          V.ThumbnailModels(),
		UpdatedAt:           V.UpdatedAt.Round(time.Second),
		URI:                 V.URI,
		Views:               V.Views,
		ViewableAt:          models.PQAsNullTime(V.ViewableAt),
		Viewable:            models.NullString(V.Viewable),
	}

	res.PreviewTemplate = V.createPreviewTemplate()
	res.ThumbnailTemplates = V.createThumbnailTemplates()
	res.IncrementViewURL, err = V.createIncrementViewURL()
	if err != nil {
		return nil, err
	}
	res.TotalLength = V.totalLength()
	res.Fps, err = V.fps()
	if err != nil {
		return nil, err
	}
	res.Resolutions, err = V.resolutions()
	if err != nil {
		return nil, err
	}
	res.Qualities, err = V.qualities()
	if err != nil {
		return nil, err
	}
	res.APIID = fmt.Sprintf("v%d", V.ID)
	res.SeekPreviewsURL = fmt.Sprintf("https://vod-storyboards.twitch.tv/%s/storyboards/%d-info.json", V.URI, V.ID)
	res.AnimatedPreviewURL = fmt.Sprintf("https://vod-storyboards.twitch.tv/%s/storyboards/%d-strip-0.jpg", V.URI, V.ID)
	res.Path = fmt.Sprintf("/videos/%d", V.ID)
	res.URL = fmt.Sprintf("https://www.twitch.tv%s", res.Path)
	if params.containShowFormats {
		res.ShowFormats, err = V.showFormats()
	}

	if err != nil {
		return nil, err
	}

	// Default viewable to 'public', so we don't start hiding new vods by accident
	if !res.Viewable.Valid || res.Viewable.String == "" {
		res.Viewable = models.NullString{Valid: true, String: "public"}
	}

	// Add `views` and `recorded_on` if necessary
	return res, nil
}

// ValuesList returns a list of the field values associated with a vod.
func (V *Vod) ValuesList() []interface{} {
	// This list follows the same order as the VodCreate fields.
	return []interface{}{
		V.bitrates,
		V.BroadcastID,
		V.BroadcastType,
		V.BroadcasterSoftware,
		V.codecs,
		V.CreatedAt,
		V.CreatedBy,
		V.Deleted,
		V.DeleteAt,
		V.Description,
		V.DescriptionHTML,
		V.displayNames,
		V.Duration,
		V.formats,
		V.Fps,
		V.Game,
		V.Language,
		V.Manifest,
		V.Offset,
		V.OwnerID,
		V.Origin,
		V.playlistPreferences,
		V.PublishedAt,
		V.Resolutions,
		V.SourceArchiveID,
		V.StartedOn,
		V.Status,
		V.TagHstore,
		V.Title,
		V.UpdatedAt,
		V.URI,
		V.Views,
		V.ViewableAt,
		V.Viewable,
		V.SelectedThumbnailID,
	}
}

func (V *Vod) status() string {
	if V.Status == models.StatusUnderReview {
		return models.StatusTranscoding
	}
	for _, s := range Statuses {
		if V.Status == s {
			return s
		}
	}
	return models.StatusRecorded
}

func (V *Vod) createPreviewTemplate() string {
	thumbnails := V.Thumbnails

	if len(thumbnails) == 0 {
		return thumbnail404
	}

	previewThumbnail := thumbnails[0]
	if V.SelectedThumbnailID.Valid {
		for _, thumb := range thumbnails {
			if thumb.ID == V.SelectedThumbnailID.Int64 {
				previewThumbnail = thumb
				break
			}
		}
	}

	return generateThumbnailURL(previewThumbnail, V.URI, V.Origin.String)
}

func (V *Vod) createThumbnailTemplates() interface{} {
	thumbnails := V.Thumbnails

	var err error
	if len(thumbnails) == 0 {
		return thumbnail404
	}

	ret := []map[string]interface{}{}
	for _, thumbnail := range thumbnails {
		template := map[string]interface{}{
			"path": thumbnail.Path,
			"url":  generateThumbnailURL(thumbnail, V.URI, V.Origin.String),
		}

		if offset := thumbnail.Offset; offset != nil {
			template["offset"] = *offset
			if err != nil {
				template["offset"] = float32(0)
			}
			template["type"] = "generated"
		} else {
			template["type"] = "custom"
		}

		ret = append(ret, template)
	}
	return ret
}

func (V *Vod) createIncrementViewURL() (string, error) {
	data := map[string]string{
		"type": "vod",
		"id":   strconv.FormatInt(V.ID, 10),
	}

	dataStr, err := json.Marshal(data)
	if err != nil {
		return "", err
	}

	values := url.Values{}
	values.Add("u", string(dataStr))

	return countessURL + "?" + values.Encode(), nil
}

func (V *Vod) totalLength() int {
	if V.Status == models.StatusRecording {
		return int(time.Since(V.CreatedAt).Seconds())
	}
	return V.Duration
}

func (V *Vod) fps() (map[string]float64, error) {
	rawFps := map[string]interface{}{}
	ret := map[string]float64{}

	if !V.Fps.Valid {
		return ret, nil
	}

	err := json.Unmarshal([]byte(V.Fps.String), &rawFps)
	if err != nil {
		return nil, err
	}

	for k, v := range rawFps {
		switch v := v.(type) {
		default:
			return nil, fmt.Errorf("unexpected type %T\n", v)
		case string:
			ret[k], err = strconv.ParseFloat(v, 64)
			if err != nil {
				return nil, err
			}
		case float64:
			ret[k] = v
		}
	}
	return ret, nil
}

func (V *Vod) resolutions() (map[string]string, error) {
	ret := map[string]string{}
	if V.Resolutions.Valid {
		err := json.Unmarshal([]byte(V.Resolutions.String), &ret)
		if err != nil {
			return nil, err
		}
	}
	return ret, nil
}

func (V *Vod) qualities() ([]string, error) {
	ret := []string{}
	if V.formats.Valid {
		err := json.Unmarshal([]byte(V.formats.String), &ret)
		if err != nil {
			return nil, err
		}
	}
	sort.StringSlice(ret).Sort()
	return ret, nil
}

func (V *Vod) showFormats() (map[string]map[string]interface{}, error) {
	ret := map[string]map[string]interface{}{}
	if !V.formats.Valid {
		return ret, nil
	}

	in := map[string]sql.NullString{
		"fps":                 V.Fps,
		"resolution":          V.Resolutions,
		"bitrate":             V.bitrates,
		"codecs":              V.codecs,
		"display_name":        V.displayNames,
		"playlist_preference": V.playlistPreferences,
	}
	videoInfo := map[string]map[string]interface{}{}

	for key, field := range in {
		buf := map[string]interface{}{}
		if field.Valid {
			d := json.NewDecoder(strings.NewReader(field.String))
			d.UseNumber()
			if err := d.Decode(&buf); err != nil {
				continue
			}
		}
		videoInfo[key] = buf
	}

	formatList := []string{}
	err := json.Unmarshal([]byte(V.formats.String), &formatList)
	if err != nil {
		return nil, err
	}

	for _, k := range formatList {
		formatMap := map[string]interface{}{}
		for infoKey, infoMap := range videoInfo {
			info, ok := infoMap[k]
			if ok {
				formatMap[infoKey] = info
			}
		}
		ret[k] = formatMap
	}

	return ret, nil
}

func generateThumbnailURL(thumbnail *VodThumbnail, uri, origin string) string {
	path := thumbnail.Path
	periodIndex := strings.Index(path, ".")
	if periodIndex == -1 {
		return thumbnail404
	}

	fileName := path[0:periodIndex]
	extension := path[periodIndex+1 : len(path)]

	customDirectory := "thumb/"
	if thumbnail.Type == "generated" {
		customDirectory = ""
	}

	prefix := ""
	if origin == "s3" {
		prefix = s3Prefix
	}

	return thumbnailHost + prefix + uri + "/" + customDirectory + fileName + "-" + size + "." + extension
}

// ReadVodRows reads the results of a query and turns it into a list of vod objects to be returned.
func ReadVodRows(rows db.Rows, queryErr error, logger errorlogger.ErrorLogger) ([]*Vod, error) {
	res := []*Vod{}
	if queryErr == sql.ErrNoRows {
		return res, nil
	}

	if queryErr != nil {
		return nil, queryErr
	}

	defer utils.CloseRows(rows, logger)

	for rows.Next() {
		vod := Vod{}
		if err := rows.Scan(
			&vod.ID,
			&vod.bitrates,
			&vod.BroadcastID,
			&vod.BroadcastType,
			&vod.BroadcasterSoftware,
			&vod.codecs,
			&vod.CreatedAt,
			&vod.CreatedBy,
			&vod.Deleted,
			&vod.DeleteAt,
			&vod.Description,
			&vod.DescriptionHTML,
			&vod.displayNames,
			&vod.Duration,
			&vod.formats,
			&vod.Fps,
			&vod.Game,
			&vod.Language,
			&vod.Manifest,
			&vod.Offset,
			&vod.OwnerID,
			&vod.Origin,
			&vod.playlistPreferences,
			&vod.PublishedAt,
			&vod.Resolutions,
			&vod.SourceArchiveID,
			&vod.StartedOn,
			&vod.Status,
			&vod.TagHstore,
			&vod.Title,
			&vod.UpdatedAt,
			&vod.URI,
			&vod.Views,
			&vod.ViewableAt,
			&vod.Viewable,
			&vod.SelectedThumbnailID,
		); err != nil {
			return nil, err
		}

		res = append(res, &vod)
	}

	return res, nil
}

// UpdateVodQuery generates the strings for the fields and values for an update sql query
func UpdateVodQuery(u models.VodUpdateInput) ([]interface{}, []interface{}, error) {
	retFields := []interface{}{}
	retValues := []interface{}{}

	if u.BroadcastID.Valid {
		retFields = append(retFields, "broadcast_id =", db.Param, ",")
		retValues = append(retValues, u.BroadcastID.Int64)
	}
	if u.BroadcastType.Valid {
		retFields = append(retFields, "broadcast_type =", db.Param, ",")
		retValues = append(retValues, u.BroadcastType.String)
	}
	if u.BroadcasterSoftware.Valid {
		retFields = append(retFields, "broadcaster_software =", db.Param, ",")
		retValues = append(retValues, u.BroadcasterSoftware.String)
	}
	if u.CreatedBy.Valid {
		retFields = append(retFields, "created_by_id =", db.Param, ",")
		retValues = append(retValues, u.CreatedBy.Int64)
	}
	if u.DeleteAt.Valid {
		retFields = append(retFields, "delete_at = to_timestamp(", db.Param, "),")
		retValues = append(retValues, u.DeleteAt.Int64)
	}
	if u.Deleted.Valid {
		retFields = append(retFields, "deleted =", db.Param, ",")
		retValues = append(retValues, u.Deleted.Bool)
	}
	if u.Description.Valid {
		retFields = append(retFields, "description =", db.Param, ",")
		retValues = append(retValues, u.Description.String)

		descriptionHTML := markdown.ConvertMarkdown(u.Description.String)
		retFields = append(retFields, "description_html =", db.Param, ",")
		retValues = append(retValues, descriptionHTML)
	}
	if u.Duration.Valid {
		retFields = append(retFields, "duration =", db.Param, ",")
		retValues = append(retValues, u.Duration.Int64)
	}
	if u.Game.Valid {
		retFields = append(retFields, "game =", db.Param, ",")
		retValues = append(retValues, u.Game.String)
	}
	if u.Language.Valid {
		retFields = append(retFields, "language =", db.Param, ",")
		retValues = append(retValues, u.Language.String)
	}
	if u.Manifest.Valid {
		retFields = append(retFields, "manifest =", db.Param, ",")
		retValues = append(retValues, u.Manifest.String)
	}
	if u.Offset.Valid {
		retFields = append(retFields, "\"offset\" =", db.Param, ",")
		retValues = append(retValues, u.Offset.Int64)
	}
	if u.Origin.Valid {
		retFields = append(retFields, "origin =", db.Param, ",")
		retValues = append(retValues, u.Origin.String)
	}
	if u.OwnerID.Valid {
		retFields = append(retFields, "owner_id =", db.Param, ",")
		retValues = append(retValues, u.OwnerID.Int64)
	}
	if u.PublishedAt.Present {
		retFields = append(retFields, "published_at =", db.Param, ",")
		retValues = append(retValues, u.PublishedAt.Time)
	}
	if u.SourceArchiveID.Valid {
		retFields = append(retFields, "source_archive_id =", db.Param, ",")
		retValues = append(retValues, u.SourceArchiveID.Int64)
	}
	if u.StartedOn.Valid {
		retFields = append(retFields, "started_on = to_timestamp(", db.Param, "),")
		retValues = append(retValues, u.StartedOn.Int64)
	}
	if u.Status.Valid {
		retFields = append(retFields, "status =", db.Param, ",")
		retValues = append(retValues, u.Status.String)
	}
	if u.Title.Valid {
		retFields = append(retFields, "title =", db.Param, ",")
		retValues = append(retValues, u.Title.String)
	}
	if u.URI.Valid {
		retFields = append(retFields, "uri =", db.Param, ",")
		retValues = append(retValues, u.URI.String)
	}
	if u.Views.Valid {
		retFields = append(retFields, "views =", db.Param, ",")
		retValues = append(retValues, u.Views.Int64)
	}
	if u.Viewable.Valid {
		retFields = append(retFields, "viewable =", db.Param, ",")
		retValues = append(retValues, u.Viewable.String)
	}
	if u.ViewableAt.Present {
		retFields = append(retFields, "viewable_at = ", db.Param, ",")
		retValues = append(retValues, u.ViewableAt.Time)
	}
	if u.TagList.Valid {
		h, err := TagListToHStore(u.TagList.String)
		if err != nil {
			return nil, nil, err
		}
		retFields = append(retFields, "tag_hstore =", db.Param, ",")
		retValues = append(retValues, h)
	}
	if u.ShowFormats != nil && len(u.ShowFormats) > 0 {
		showFormatsVod := Vod{}

		err := ApplyShowFormats(&models.Vod{ShowFormats: u.ShowFormats}, &showFormatsVod)
		if err != nil {
			return nil, nil, err
		}
		if showFormatsVod.formats.Valid {
			retFields = append(retFields, "formats =", db.Param, ",")
			retValues = append(retValues, showFormatsVod.formats.String)
		}
		if showFormatsVod.bitrates.Valid {
			retFields = append(retFields, "bitrates =", db.Param, ",")
			retValues = append(retValues, showFormatsVod.bitrates.String)
		}
		if showFormatsVod.codecs.Valid {
			retFields = append(retFields, "codecs =", db.Param, ",")
			retValues = append(retValues, showFormatsVod.codecs.String)
		}
		if showFormatsVod.displayNames.Valid {
			retFields = append(retFields, "display_names =", db.Param, ",")
			retValues = append(retValues, showFormatsVod.displayNames.String)
		}
		if showFormatsVod.playlistPreferences.Valid {
			retFields = append(retFields, "playlist_preferences =", db.Param, ",")
			retValues = append(retValues, showFormatsVod.playlistPreferences.String)
		}
		if showFormatsVod.Resolutions.Valid {
			retFields = append(retFields, "resolutions =", db.Param, ",")
			retValues = append(retValues, showFormatsVod.Resolutions.String)
		}
		if showFormatsVod.Fps.Valid {
			retFields = append(retFields, "fps =", db.Param, ",")
			retValues = append(retValues, showFormatsVod.Fps.String)
		}
	}

	retFields = append(retFields, "updated_at =", db.Param)
	retValues = append(retValues, u.UpdatedAt)

	return retFields, retValues, nil
}
