package models

import (
	"database/sql"
	"fmt"
	"strings"
	"time"

	"code.justin.tv/chat/db"
	"code.justin.tv/common/golibs/errorlogger"
	dsutils "code.justin.tv/vod/vinyl/datastore/vinyldb/utils"
	"code.justin.tv/vod/vinyl/utils"

	"code.justin.tv/vod/vinyl/models"
	"github.com/lib/pq"
)

// Constants associated with the vod appeals table and its fields.
var (
	VodAppealsTableName   = "vod_appeals"
	VodAppealCreateFields = []string{
		"resolved_at", "created_at", "updated_at",
		"vod_id", "priority", "full_name", "street_address_1",
		"street_address_2", "city", "state", "zipcode",
		"country"}
	VodAppealFullyQualifiedCreateFields = utils.FullyQualifyAttributeName(
		VodAppealCreateFields, VodAppealsTableName)

	VodAppealFields = append(utils.FullyQualifyAttributeName([]string{"id"}, VodAppealsTableName), VodAppealFullyQualifiedCreateFields...)
)

// FetchVODAppeals returns a string used to query for all the fields of a vod appeal.
func FetchVODAppeals() string {
	return "SELECT " + strings.Join(VodAppealFields, ",") + " FROM " + VodAppealsTableName +
		" JOIN " + fmt.Sprintf("%s ON %s.id=%s.vod_id", VodTableName, VodTableName, VodAppealsTableName)
}

// VodAppeal represents a VinylDB vod appeal object.
type VodAppeal struct {
	VodAppealID    int64
	ResolvedAt     pq.NullTime
	CreatedAt      time.Time
	UpdatedAt      time.Time
	VodID          int64
	Priority       bool
	FullName       string
	StreetAddress1 string
	StreetAddress2 string
	City           string
	State          string
	Zipcode        string
	Country        string
}

// FromVinylVodAppeal converts a Vinyl vod object into a VinylDB vod object to interact with the database.
func FromVinylVodAppeal(vinylVodAppeal *models.VodAppeal) (*VodAppeal, error) {
	return &VodAppeal{
		VodAppealID:    vinylVodAppeal.VodAppealID,
		ResolvedAt:     vinylVodAppeal.ResolvedAt.AsPQ(),
		CreatedAt:      vinylVodAppeal.CreatedAt,
		UpdatedAt:      vinylVodAppeal.UpdatedAt,
		VodID:          vinylVodAppeal.VodID,
		Priority:       vinylVodAppeal.Priority,
		FullName:       vinylVodAppeal.FullName,
		StreetAddress1: vinylVodAppeal.StreetAddress1,
		StreetAddress2: vinylVodAppeal.StreetAddress2,
		City:           vinylVodAppeal.City,
		State:          vinylVodAppeal.State,
		Zipcode:        vinylVodAppeal.Zipcode,
		Country:        vinylVodAppeal.Country,
	}, nil
}

// AsVinylVodAppeal converts a VinylDB vod object into a Vinyl vod object to be returned.
func (V *VodAppeal) AsVinylVodAppeal() (*models.VodAppeal, error) {
	resolvedAt := pq.NullTime{Valid: V.ResolvedAt.Valid}
	if V.ResolvedAt.Valid {
		resolvedAt.Time = V.ResolvedAt.Time.Round(time.Second)
	}

	return &models.VodAppeal{
		VodAppealID:    V.VodAppealID,
		ResolvedAt:     models.PQAsNullTime(resolvedAt),
		CreatedAt:      V.CreatedAt.Round(time.Second),
		UpdatedAt:      V.UpdatedAt.Round(time.Second),
		VodID:          V.VodID,
		Priority:       V.Priority,
		FullName:       V.FullName,
		StreetAddress1: V.StreetAddress1,
		StreetAddress2: V.StreetAddress2,
		City:           V.City,
		State:          V.State,
		Zipcode:        V.Zipcode,
		Country:        V.Country,
	}, nil
}

// ValuesList returns a list of the field values associated with a vod.
func (V *VodAppeal) ValuesList() []interface{} {
	return []interface{}{
		V.ResolvedAt,
		V.CreatedAt,
		V.UpdatedAt,
		V.VodID,
		V.Priority,
		V.FullName,
		V.StreetAddress1,
		V.StreetAddress2,
		V.City,
		V.State,
		V.Zipcode,
		V.Country,
	}
}

// ReadVodAppealRows converts the results of a query into a list of vod objects to be returned.
func ReadVodAppealRows(rows db.Rows, queryErr error, logger errorlogger.ErrorLogger) ([]*VodAppeal, error) {
	res := []*VodAppeal{}
	if queryErr == sql.ErrNoRows {
		return res, nil
	}

	if queryErr != nil {
		return nil, queryErr
	}

	defer dsutils.CloseRows(rows, logger)

	for rows.Next() {
		vodAppeal := VodAppeal{}
		if err := rows.Scan(
			&vodAppeal.VodAppealID,
			&vodAppeal.ResolvedAt,
			&vodAppeal.CreatedAt,
			&vodAppeal.UpdatedAt,
			&vodAppeal.VodID,
			&vodAppeal.Priority,
			&vodAppeal.FullName,
			&vodAppeal.StreetAddress1,
			&vodAppeal.StreetAddress2,
			&vodAppeal.City,
			&vodAppeal.State,
			&vodAppeal.Zipcode,
			&vodAppeal.Country,
		); err != nil {
			return res, nil
		}

		res = append(res, &vodAppeal)
	}

	return res, nil
}
