package models

import (
	"database/sql"
	"strings"
	"time"

	"github.com/lib/pq"

	"code.justin.tv/chat/db"
	"code.justin.tv/common/golibs/errorlogger"
	"code.justin.tv/vod/vinyl/datastore/vinyldb/utils"
	"code.justin.tv/vod/vinyl/models"
)

// Constants used to keep track of the notification settings table and its fields.
var (
	VodNotificationSettingsUpdateFields = []string{
		"custom_text", "enabled", "sent_at", "updated_at",
	}
	VodNotificationSettingsCreateFields = append([]string{"vod_id", "type", "created_at"}, VodNotificationSettingsUpdateFields...)
	VodNotificationSettingsFields       = append([]string{"id"}, VodNotificationSettingsCreateFields...)
	VodNotificationSettingsTableName    = "vod_notification_settings"
)

// VodNotificationSettings is the representation for a VinylDB notification settings row.
type VodNotificationSettings struct {
	ID         int64
	VodID      int64
	Type       string
	CustomText sql.NullString
	Enabled    bool
	SentAt     pq.NullTime
	CreatedAt  time.Time
	UpdatedAt  time.Time
}

// FetchVodNotificationSettingsQuery returns a string used to grab all fields from the notification settings table.
func FetchVodNotificationSettingsQuery() string {
	return "SELECT " + strings.Join(VodNotificationSettingsFields, ",") + " FROM " + VodNotificationSettingsTableName
}

// ReadVodNotificationSettingsRows converts the results of a query into a list of VodNotificationSettings objects.
func ReadVodNotificationSettingsRows(rows db.Rows, queryErr error, logger errorlogger.ErrorLogger) ([]*VodNotificationSettings, error) {
	res := []*VodNotificationSettings{}

	if queryErr == sql.ErrNoRows {
		return res, nil
	}

	if queryErr != nil {
		return nil, queryErr
	}

	defer utils.CloseRows(rows, logger)

	for rows.Next() {
		notificationSettings := VodNotificationSettings{}
		if err := rows.Scan(
			&notificationSettings.ID,
			&notificationSettings.VodID,
			&notificationSettings.Type,
			&notificationSettings.CreatedAt,
			&notificationSettings.CustomText,
			&notificationSettings.Enabled,
			&notificationSettings.SentAt,
			&notificationSettings.UpdatedAt,
		); err != nil {
			return nil, err
		}

		res = append(res, &notificationSettings)
	}
	return res, nil
}

// AsVinylVodNotificationSettings converts a VodNotificationSettings object into a VinylVodNotificationSettings object to be returned.
func (N *VodNotificationSettings) AsVinylVodNotificationSettings() (*models.VodNotificationSettings, error) {
	return &models.VodNotificationSettings{
		ID:         N.ID,
		VodID:      N.VodID,
		Type:       N.Type,
		CustomText: models.NullString(N.CustomText),
		Enabled:    N.Enabled,
		SentAt:     models.PQAsNullTime(N.SentAt),
		CreatedAt:  N.CreatedAt,
		UpdatedAt:  N.UpdatedAt,
	}, nil
}

// ValuesList returns a list of the field values associated with a VodNotificationSettings.
func (N *VodNotificationSettings) ValuesList() []interface{} {
	return []interface{}{
		N.VodID,
		N.Type,
		N.CreatedAt,
		N.CustomText,
		N.Enabled,
		N.SentAt,
		N.UpdatedAt,
	}
}

// FromVinylVodNotificationSettings converts a Vinyl VodNotificationSettings object into a VodNotificationSettings to be used in the vinyldb package.
func FromVinylVodNotificationSettings(vinylNotificationSettings *models.VodNotificationSettings) (*VodNotificationSettings, error) {
	return &VodNotificationSettings{
		ID:         vinylNotificationSettings.ID,
		VodID:      vinylNotificationSettings.VodID,
		Type:       vinylNotificationSettings.Type,
		CustomText: sql.NullString(vinylNotificationSettings.CustomText),
		Enabled:    vinylNotificationSettings.Enabled,
		SentAt:     vinylNotificationSettings.SentAt.AsPQ(),
		CreatedAt:  vinylNotificationSettings.CreatedAt.Round(time.Second),
		UpdatedAt:  vinylNotificationSettings.UpdatedAt.Round(time.Second),
	}, nil
}

// FromVinylVodNotificationSettingsInput converts a Vinyl VodNotificationSettingsInput object into a VodNotificationSettings to be used in the vinyldb package.
func FromVinylVodNotificationSettingsInput(vinylNotificationSettingsInput *models.VodNotificationSettingsInput, vodID int64, notificationType string) (*VodNotificationSettings, error) {
	return &VodNotificationSettings{
		VodID:      vodID,
		Type:       notificationType,
		CustomText: sql.NullString(vinylNotificationSettingsInput.CustomText),
		Enabled:    vinylNotificationSettingsInput.Enabled,
		SentAt:     vinylNotificationSettingsInput.SentAt.AsPQ(),
	}, nil
}

// UpdateVodNotificationSettingsQuery generates the strings for the fields and values for an update sql query
func UpdateVodNotificationSettingsQuery(u models.VodNotificationSettingsInput) ([]interface{}, []interface{}, error) {
	retFields := []interface{}{}
	retValues := []interface{}{}

	if u.CustomText.Valid {
		retFields = append(retFields, "custom_text =", db.Param, ",")
		retValues = append(retValues, u.CustomText.String)
	}

	if u.SentAt.Present {
		retFields = append(retFields, "sent_at =", db.Param, ",")
		retValues = append(retValues, u.SentAt.Time)
	}

	retFields = append(retFields, "enabled =", db.Param, ",")
	retValues = append(retValues, u.Enabled)

	retFields = append(retFields, "updated_at =", db.Param)
	retValues = append(retValues, time.Now().UTC().Round(time.Second))

	return retFields, retValues, nil
}
