package models

import (
	"database/sql"
	"testing"
	"time"

	"code.justin.tv/chat/db"
	"code.justin.tv/vod/vinyl/errors"
	"code.justin.tv/vod/vinyl/models"

	"github.com/lib/pq/hstore"
	. "github.com/smartystreets/goconvey/convey"
)

func TestCreateImageTemplates(t *testing.T) {
	Convey("When creating", t, func() {
		vod := makeTestVod()

		Convey("preview templates", func() {
			Convey("Should create the correct thumbnail URI", func() {
				vod.Thumbnails = VodThumbnails{
					&VodThumbnail{
						Path: "thumb0.jpg",
					},
				}
				res := vod.createPreviewTemplate()
				So(res, ShouldEqual, "https://static-cdn.jtvnw.net/s3_vods/v1/AUTH_system/vods_aef7/coolest_vod_in_the_world/thumb/thumb0-%{width}x%{height}.jpg")
			})

			Convey("When thumbnails are empty, should return 404 URI", func() {
				res := vod.createPreviewTemplate()
				So(res, ShouldEqual, thumbnail404)
			})

		})
		Convey("thumbnail templates", func() {
			Convey("valid", func() {
				float1 := float32(24914.066000000003)
				float2 := float32(19218.066000000003)
				vod.Thumbnails = VodThumbnails{
					&VodThumbnail{
						Path:   "thumb/thumb0.jpg",
						Offset: &float1,
						Type:   "generated",
					},
					&VodThumbnail{
						Path:   "thumb/thumb1.jpg",
						Offset: &float2,
						Type:   "generated",
					},
					&VodThumbnail{
						Path: "customthumb.jpg",
					},
				}
				res := vod.createThumbnailTemplates()
				So(res, ShouldResemble, []map[string]interface{}{
					map[string]interface{}{
						"path":   "thumb/thumb0.jpg",
						"url":    "https://static-cdn.jtvnw.net/s3_vods/v1/AUTH_system/vods_aef7/coolest_vod_in_the_world/thumb/thumb0-%{width}x%{height}.jpg",
						"offset": float32(24914.066000000003),
						"type":   "generated",
					},
					map[string]interface{}{
						"path":   "thumb/thumb1.jpg",
						"url":    "https://static-cdn.jtvnw.net/s3_vods/v1/AUTH_system/vods_aef7/coolest_vod_in_the_world/thumb/thumb1-%{width}x%{height}.jpg",
						"offset": float32(19218.066000000003),
						"type":   "generated",
					},
					map[string]interface{}{
						"path": "customthumb.jpg",
						"url":  "https://static-cdn.jtvnw.net/s3_vods/v1/AUTH_system/vods_aef7/coolest_vod_in_the_world/thumb/customthumb-%{width}x%{height}.jpg",
						"type": "custom",
					},
				})
			})
			Convey("null", func() {
				res := vod.createThumbnailTemplates()
				So(res, ShouldEqual, thumbnail404)
			})
		})
	})
}

func TestCreateIncrementViewURL(t *testing.T) {
	Convey("When creating the increment view URL", t, func() {
		vod := makeTestVod()

		Convey("Should create the URL successfully", func() {
			res, err := vod.createIncrementViewURL()
			So(err, ShouldBeNil)
			So(res, ShouldEqual, "https://countess.twitch.tv/ping.gif?u=%7B%22id%22%3A%221%22%2C%22type%22%3A%22vod%22%7D")
		})
	})
}

func TestTotalLength(t *testing.T) {
	Convey("When computing total length", t, func() {
		vod := makeTestVod()
		Convey("When status is recording", func() {
			vod.Status = models.StatusRecording
			res := vod.totalLength()
			So(res, ShouldEqual, 0)
		})

		Convey("When status is not recording", func() {
			vod.Status = models.StatusRecorded
			res := vod.totalLength()
			So(res, ShouldEqual, vod.Duration)
		})
	})
}

func TestStatus(t *testing.T) {
	Convey("Status", t, func() {
		vod := makeTestVod()
		Convey("When status is in the list", func() {
			vod.Status = models.StatusRecording
			res := vod.status()
			So(res, ShouldEqual, models.StatusRecording)
		})

		Convey("When status is invalid", func() {
			vod.Status = "oogabooga"
			res := vod.status()
			So(res, ShouldEqual, models.StatusRecorded)
		})

		Convey("When status is under_review", func() {
			vod.Status = models.StatusUnderReview
			res := vod.status()
			So(res, ShouldEqual, models.StatusTranscoding)
		})
	})
}

func TestFps(t *testing.T) {
	Convey("When computing fps", t, func() {
		vod := makeTestVod()
		expected := map[string]float64{
			"audio_only": 0,
			"medium":     29.999241465379235,
		}
		Convey("containing float64", func() {
			vod.Fps = sql.NullString{Valid: true, String: `{"audio_only":0,"medium":29.999241465379235}`}
			res, err := vod.fps()
			So(res, ShouldResemble, expected)
			So(err, ShouldBeNil)
		})

		Convey("When fps contains string values", func() {
			vod.Fps = sql.NullString{Valid: true, String: `{"audio_only":"0","medium":"29.999241465379235"}`}
			res, err := vod.fps()
			So(res, ShouldResemble, expected)
			So(err, ShouldBeNil)
		})

		Convey("When fps contains non-numbers", func() {
			vod.Fps = sql.NullString{Valid: true, String: `{"audio_only":0,"medium":"bananas"}`}
			res, err := vod.fps()
			So(res, ShouldBeNil)
			So(err, ShouldNotBeNil)
		})

		Convey("When fps isnull", func() {
			vod.Fps = sql.NullString{Valid: false, String: ``}
			res, err := vod.fps()
			So(res, ShouldBeEmpty)
			So(err, ShouldBeNil)
		})
	})
}

func TestQualities(t *testing.T) {
	Convey("When extracting the qualities", t, func() {
		vod := makeTestVod()
		Convey("Should have different qualities after parsing", func() {
			res, err := vod.qualities()
			So(err, ShouldBeNil)
			So(res, ShouldNotBeEmpty)
			So(res[0], ShouldEqual, "audio_only")
		})
	})

}
func TestShowFormats(t *testing.T) {
	Convey("When creating the ShowFormats map", t, func() {
		vod := makeTestVod()
		Convey("Should have a fully populated map based on the raw fields", func() {
			res, err := vod.showFormats()
			So(err, ShouldBeNil)
			So(res, ShouldNotBeEmpty)
			So(res["medium"]["resolution"], ShouldEqual, "852x480")
		})
	})
}

func TestUpdateVodQuery(t *testing.T) {
	var u models.VodUpdateInput
	now := time.Now().UTC().Round(time.Second)
	Convey("When creating fields and values for update query", t, func() {

		Convey("empty", func() {
			u = models.VodUpdateInput{UpdatedAt: now}
			retFields, retValues, err := UpdateVodQuery(u)

			So(err, ShouldBeNil)
			So(retFields, ShouldResemble, []interface{}{"updated_at =", db.Param})
			So(retValues, ShouldResemble, []interface{}{now})
		})
		Convey("basic field", func() {
			title := "my title"
			u = models.VodUpdateInput{UpdatedAt: now, Title: models.NullString{String: title, Valid: true}}
			retFields, retValues, err := UpdateVodQuery(u)

			So(err, ShouldBeNil)
			So(retFields, ShouldResemble, []interface{}{"title =", db.Param, ",", "updated_at =", db.Param})
			So(retValues, ShouldResemble, []interface{}{title, now})
		})
		Convey("tag list", func() {
			tagList := "one,   two,three, \nfour"
			u = models.VodUpdateInput{UpdatedAt: now, TagList: models.NullString{String: tagList, Valid: true}}
			retFields, retValues, err := UpdateVodQuery(u)

			So(err, ShouldBeNil)
			So(retFields, ShouldResemble, []interface{}{"tag_hstore =", db.Param, ",", "updated_at =", db.Param})
			So(retValues, ShouldResemble, []interface{}{hstore.Hstore{
				Map: map[string]sql.NullString{
					"one":   sql.NullString{Valid: false},
					"two":   sql.NullString{Valid: false},
					"three": sql.NullString{Valid: false},
					"four":  sql.NullString{Valid: false},
				},
			}, now})
		})
		Convey("show formats", func() {
			var showFormats map[string]map[string]interface{}
			Convey("empty", func() {
				showFormats = map[string]map[string]interface{}{}
				u = models.VodUpdateInput{UpdatedAt: now, ShowFormats: showFormats}
				retFields, retValues, err := UpdateVodQuery(u)

				So(err, ShouldBeNil)
				So(retFields, ShouldResemble, []interface{}{"updated_at =", db.Param})
				So(retValues, ShouldResemble, []interface{}{now})
			})
			Convey("with some fields present", func() {
				showFormats = map[string]map[string]interface{}{
					"audio_only": map[string]interface{}{
						"playlist_preference": 1,
						"display_name":        "Audio Only",
						"bitrate":             190729,
						"fps":                 0.0,
						"codecs":              "mp4a.40.2",
					},
					"chunked": map[string]interface{}{
						"playlist_preference": 0,
						"display_name":        "Source",
						"bitrate":             1910583,
						"fps":                 22.02391691282793,
						"codecs":              "avc1.64001F, mp4a.40.2",
						"resolution":          "1280x720"},
				}

				u = models.VodUpdateInput{UpdatedAt: now, ShowFormats: showFormats}
				retFields, retValues, err := UpdateVodQuery(u)
				expectedFields := []interface{}{
					"formats =", db.Param, ",",
					"bitrates =", db.Param, ",",
					"codecs =", db.Param, ",",
					"display_names =", db.Param, ",",
					"playlist_preferences =", db.Param, ",",
					"resolutions =", db.Param, ",",
					"fps =", db.Param, ",",
					"updated_at =", db.Param,
				}
				expectedValues := []interface{}{
					`["audio_only","chunked"]`,
					`{"audio_only":190729,"chunked":1910583}`,
					`{"audio_only":"mp4a.40.2","chunked":"avc1.64001F, mp4a.40.2"}`,
					`{"audio_only":"Audio Only","chunked":"Source"}`,
					`{"audio_only":1,"chunked":0}`,
					`{"chunked":"1280x720"}`,
					`{"audio_only":0,"chunked":22.02391691282793}`,
					now,
				}

				So(err, ShouldBeNil)
				So(retFields, ShouldResemble, expectedFields)
				// Avoid comparing `formats` values to avoid nondeterministic test
				// This issue occurs `formats` is generated by iterating over a hash,
				// and iterating over a hash in go occurs in random order, so possible
				// values are `["audio_only","chunked"]` or `["chunked","audio_only"]`
				So(retValues[1:], ShouldResemble, expectedValues[1:])
			})
		})
	})
}

func TestTagListToHStore(t *testing.T) {
	Convey("basic", t, func() {
		expected := hstore.Hstore{Map: map[string]sql.NullString{"tag1": sql.NullString{Valid: false}}}
		res, err := TagListToHStore("tag1")
		So(expected, ShouldResemble, res)
		So(err, ShouldBeNil)
	})
	Convey("multiple tags", t, func() {
		expected := hstore.Hstore{Map: map[string]sql.NullString{
			"tag1": sql.NullString{Valid: false},
			"tag2": sql.NullString{Valid: false},
		}}
		res, err := TagListToHStore("tag1, tag2")
		So(expected, ShouldResemble, res)
		So(err, ShouldBeNil)
	})
	Convey("one tag is too long", t, func() {
		expected := hstore.Hstore{}
		res, err := TagListToHStore("tag1, fox jumps over the lazy dogthe quick brown fox jumps over the lazy dogthe quick brown fox jumps over the lazy dogthe quick brown fox jumps over the lazy dogthe quick brown fox jumps over the lazy dogthe quick brown fox jumps over the lazy dogthe quick brown fox jumps over the lazy dogthe quick brown fox jumps over the lazy dogthe quick brown fox jumps over the lazy dogthe quick brown fox jumps over the lazy dog")
		So(res, ShouldResemble, expected)
		So(err, ShouldResemble, errors.PerTagTooLongError{})
	})
	Convey("entire tag field is too long", t, func() {
		expected := hstore.Hstore{}
		res, err := TagListToHStore(`
			Is, this, the, real, life?., Is, this, just, fantasy?., Caught, in, a, landslide., No, escape, from, reality., Open, your, eyes., Look, up, to, the, skies, and, see., I'm, just, a, poor, boy,, I, need, no, sympathy., Because, I'm, easy, come,, easy, go., A, little, high,, little, low., Anyway, the, wind, blows,, doesn't, really, matter, to, me,, to, me.,

Mama,, just, killed, a, man., Put, a, gun, against, his, head., Pulled, my, trigger,, now, he's, dead., Mama,, life, had, just, begun., But, now, I've, gone, and, thrown, it, all, away., Mama,, ooo., Didn't, mean, to, make, you, cry., If, I'm, not, back, again, this, time, tomorrow., Carry, on,, carry, on,, as, if, nothing, really, matters.,

Too, late,, my, time, has, come., Sends, shivers, down, my, spine., Body's, aching, all, the, time., Goodbye, everybody, I've, got, to, go., Gotta, leave, you, all, behind, and, face, the, truth., Mama,, ooo, (anyway, the, wind, blows)., I, don't, want, to, die., I, sometimes, wish, I'd, never, been, born, at, all.,

I, see, a, little, silhouetto, of, a, man., Scaramouch,, scaramouch, will, you, do, the, fandango., Thunderbolt, and, lightning, very, very, frightening, me., Gallileo,, Gallileo,., Gallileo,, Gallileo,., Gallileo, Figaro, -, magnifico.,

But, I'm, just, a, poor, boy, and, nobody, loves, me., He's, just, a, poor, boy, from, a, poor, family., Spare, him, his, life, from, this, monstrosity., Easy, come, easy, go, will, you, let, me, go., Bismillah!, No, we, will, not, let, you, go, -, let, him, go., Bismillah!, We, will, not, let, you, go, -, let, him, go., Bismillah!, We, will, not, let, you, go, let, me, go., Will, not, let, you, go, let, me, go, (never)., Never, let, you, go, let, me, go., Never, let, me, go, ooo., No,, no,, no,, no,, no,, no,, no., Oh, mama, mia,, mama, mia,, mama, mia, let, me, go., Beelzebub, has, a, devil, put, aside, for, me., For, me., For, me.,

So, you, think, you, can, stone, me, and, spit, in, my, eye., So, you, think, you, can, love, me, and, leave, me, to, die., Oh, baby,, can't, do, this, to, me, baby., Just, gotta, get, out, just, gotta, get, right, outta, here.,

Ooh, yeah,, ooh, yeah., Nothing, really, matters., Anyone, can, see., Nothing, really, matters, nothing, really, matters, to, me.,

Anyway, the, wind, blows
			`)
		So(res, ShouldResemble, expected)
		So(err, ShouldResemble, errors.TagFieldTooLongError{})
	})
}

func makeTestVod() *Vod {
	return &Vod{
		BroadcastID:         12345,
		BroadcastType:       "archive",
		BroadcasterSoftware: sql.NullString{Valid: true, String: "obs"},
		CreatedAt:           time.Now(),
		Deleted:             sql.NullBool{Valid: false, Bool: false},
		Description:         sql.NullString{Valid: true, String: "This is not the coolest VOD in the world. This is just a tribute"},
		Duration:            60,
		Fps:                 sql.NullString{Valid: true, String: `{"audio_only":0,"medium":29.999241465379235,"mobile":29.999241465379235,"high":29.999241465379235,"low":29.999241465379235,"chunked":29.99904566398}`},
		Game:                sql.NullString{Valid: true, String: "Dota 2"},
		ID:                  1,
		Language:            sql.NullString{Valid: true, String: "nl"},
		Manifest:            sql.NullString{Valid: true, String: "index-elegiggle.m3u8"},
		Offset:              0,
		Origin:              sql.NullString{Valid: true, String: "s3"},
		OwnerID:             644,
		Resolutions:         sql.NullString{Valid: true, String: `{"medium":"852x480","mobile":"400x226","high":"1280x720","low":"640x360","chunked":"1280x720"}`},
		Status:              models.StatusRecorded,
		TagHstore:           hstore.Hstore{Map: nil},
		Title:               sql.NullString{Valid: true, String: "Coolest VOD"},
		UpdatedAt:           time.Now(),
		URI:                 "v1/AUTH_system/vods_aef7/coolest_vod_in_the_world",
		Views:               456,

		bitrates:            sql.NullString{Valid: true, String: `{"audio_only":200609,"medium":835998,"mobile":296938,"high":1530246,"low":582257,"chunked":2740420}`},
		codecs:              sql.NullString{Valid: true, String: `{"audio_only":"mp4a.40.2","medium":"avc1.42C01E,mp4a.40.2","mobile":"avc1.42C00D,mp4a.40.2","high":"avc1.42C01F,mp4a.40.2","low":"avc1.42C01E,mp4a.40.2","chunked":"avc1.4D401F,mp4a.40.2"}`},
		displayNames:        sql.NullString{Valid: true, String: `{"audio_only":"Audio Only","medium":"Medium","mobile":"Mobile","high":"High","low":"Low","chunked":"Source"}`},
		formats:             sql.NullString{Valid: true, String: `["audio_only","medium","mobile","high","low","chunked"]`},
		playlistPreferences: sql.NullString{Valid: true, String: `{"audio_only":5,"medium":2,"mobile":4,"high":1,"low":3,"chunked":0}`},
	}
}
