package vinyldb

import (
	"time"

	"golang.org/x/net/context"

	"code.justin.tv/chat/db"
	"code.justin.tv/vod/vinyl/datastore/vinyldb/models"
	"code.justin.tv/vod/vinyl/errors"
)

// ResolveTrackAppeal marks a track appeal to be resolved
// and unmutes the corresponding AMR if it is approved.
// Returns the amrID and vodID information to post to Usher if needed.
func (b *Backend) ResolveTrackAppeal(ctx context.Context, appealID int64, approve bool) (int64, int64, error) {
	now := time.Now().UTC().Round(time.Second)

	query := db.BuildQuery("UPDATE", models.TrackAppealsTableName,
		"SET updated_at =", db.Param, ", resolved_at =", db.Param,
		"WHERE id = ", db.Param, "RETURNING audible_magic_response_id")

	row := b.conn.QueryRow(ctx, "resolve_track_appeal", query, now, now, appealID)

	var amrID int64
	err := row.Scan(&amrID)
	if err == db.ErrNoRows {
		return 0, 0, errors.NotFoundError{Type: "Track appeal", ID: int(appealID)}
	}
	if err != nil {
		return 0, 0, err
	}

	var vodID int64
	if approve {
		updateFields := map[string]interface{}{
			"unmuted_at": now,
			"updated_at": now,
		}

		dAMR, err := b.UpdateAMR(ctx, int(amrID), updateFields)
		if err != nil {
			return 0, 0, err
		}

		updatedAMR, err := dAMR.AsVinylAMR()
		if err != nil {
			return 0, 0, err
		}

		vodID = updatedAMR.VodID
	}

	return amrID, vodID, nil
}
