package vinyldb

import (
	"testing"
	"time"

	"code.justin.tv/vod/vinyl/datastore/vinyldb/models"

	sqlmock "github.com/DATA-DOG/go-sqlmock"
	. "github.com/smartystreets/goconvey/convey"

	"golang.org/x/net/context"
)

func TestResolveTrackAppeal(t *testing.T) {
	Convey("ResolveTrackAppeal", t, func() {
		mock, backend := Setup()

		Convey("only updates the resolved time if the track is rejected", func() {
			appealID := int64(1238)
			query := `UPDATE track_appeals SET updated_at = \$1 , resolved_at = \$2 WHERE id = \$3 RETURNING audible_magic_response_id`

			fakeResult := sqlmock.NewRows([]string{"audible_magic_response_id"})
			fakeResult.AddRow(1)
			mock.ExpectQuery(query).WithArgs(AnyTime{}, AnyTime{}, appealID).WillReturnRows(fakeResult)

			amrID, _, err := backend.ResolveTrackAppeal(context.Background(), appealID, false)
			So(mock.ExpectationsWereMet(), ShouldBeNil)
			So(err, ShouldBeNil)
			So(amrID, ShouldEqual, 1)
		})

		Convey("returns an error if the track appeal wasn't found", func() {
			appealID := int64(1238)
			query := `UPDATE track_appeals SET updated_at = \$1 , resolved_at = \$2 WHERE id = \$3 RETURNING audible_magic_response_id`

			fakeResult := sqlmock.NewRows([]string{"audible_magic_response_id"})
			mock.ExpectQuery(query).WithArgs(AnyTime{}, AnyTime{}, appealID).WillReturnRows(fakeResult)

			_, _, err := backend.ResolveTrackAppeal(context.Background(), appealID, false)
			So(mock.ExpectationsWereMet(), ShouldBeNil)
			So(err.Error(), ShouldContainSubstring, "Track appeal with ID 1238 not found")
		})

		Convey("returns an error if the base AMR is not found", func() {
			appealID := int64(1238)
			query := `UPDATE track_appeals SET updated_at = \$1 , resolved_at = \$2 WHERE id = \$3 RETURNING audible_magic_response_id`

			fakeResult := sqlmock.NewRows([]string{"audible_magic_response_id"})
			fakeResult.AddRow(1)
			mock.ExpectQuery(query).WithArgs(AnyTime{}, AnyTime{}, appealID).WillReturnRows(fakeResult)

			query = `UPDATE audible_magic_responses SET ` +
				`(updated_at|unmuted_at) = \$1 , (updated_at|unmuted_at) = \$2 ` +
				`WHERE id = \$3 RETURNING (.+)`
			amrResult := sqlmock.NewRows(models.AMRFields)
			mock.ExpectQuery(query).WithArgs(AnyTime{}, AnyTime{}, 1).WillReturnRows(amrResult)

			_, _, err := backend.ResolveTrackAppeal(context.Background(), appealID, true)
			So(mock.ExpectationsWereMet(), ShouldBeNil)
			So(err.Error(), ShouldContainSubstring, "AMR with ID 1 not found")
		})

		Convey("modifies the base AMR properly when found", func() {
			appealID := int64(1238)
			query := `UPDATE track_appeals SET updated_at = \$1 , resolved_at = \$2 WHERE id = \$3 RETURNING audible_magic_response_id`

			fakeResult := sqlmock.NewRows([]string{"audible_magic_response_id"})
			fakeResult.AddRow(1)
			mock.ExpectQuery(query).WithArgs(AnyTime{}, AnyTime{}, appealID).WillReturnRows(fakeResult)

			now := time.Now()
			query = `UPDATE audible_magic_responses SET ` +
				`(updated_at|unmuted_at) = \$1 , (updated_at|unmuted_at) = \$2 ` +
				`WHERE id = \$3 RETURNING (.+)`
			amrResult := sqlmock.NewRows(models.AMRFields)
			amrResult.AddRow(2, 777, nil, nil, nil, nil, nil, nil, nil, true, nil, nil, nil, nil, nil, nil, now, now, nil, nil, nil)
			mock.ExpectQuery(query).WithArgs(AnyTime{}, AnyTime{}, 1).WillReturnRows(amrResult)

			amrID, vodID, err := backend.ResolveTrackAppeal(context.Background(), appealID, true)
			So(mock.ExpectationsWereMet(), ShouldBeNil)
			So(err, ShouldBeNil)
			So(amrID, ShouldEqual, 1)
			So(vodID, ShouldEqual, 777)
		})
	})
}
