package vinyldb

import (
	"fmt"
	"time"

	"golang.org/x/net/context"

	"code.justin.tv/chat/db"
	"code.justin.tv/vod/vinyl/datastore/vinyldb/models"
	"code.justin.tv/vod/vinyl/errors"
	edgemodels "code.justin.tv/vod/vinyl/models"
)

// ResolveVodAppeal marks a vod appeal as resolved.
func (b *Backend) ResolveVodAppeal(ctx context.Context, appealID int64) (int64, error) {
	// Ideally we would count the number of unresolved track appeals. However, the database is currently in a bad state where
	// it is possible for a track appeal to have references to AMRs with a different vod_id than that of the track appeal.
	// Also, we check if the AMR.unmuted_at is NULL because it is also possible for multiple track appeals within a vod appeal
	// to point to the same AMR, with only one of those track appeals being set to resolved.
	// We check if track_appeals.resolved_at is NULL because the AMR takes a while to be unmuted after a track is resolved.
	// This will be solved when we take a look at vinyl's vod appeals schema for a rewrite.
	// feelsbadman
	query := db.BuildQuery("SELECT COUNT(1) FROM", models.TrackAppealsTableName,
		"JOIN", models.AMRTableName, "ON", fmt.Sprintf("%v.id=%v.audible_magic_response_id", models.AMRTableName, models.TrackAppealsTableName),
		"JOIN", models.VodAppealsTableName, "ON", fmt.Sprintf("%v.id=%v.vod_appeal_id", models.VodAppealsTableName, models.TrackAppealsTableName), "AND", fmt.Sprintf("%v.vod_id=%v.vod_id", models.VodAppealsTableName, models.AMRTableName),
		"WHERE vod_appeal_id =", db.Param, fmt.Sprintf("AND %v.unmuted_at is NULL AND %v.resolved_at is NULL", models.AMRTableName, models.TrackAppealsTableName))

	row := b.conn.QueryRow(ctx, "count_unresolved_track_appeals", query, appealID)
	var count int64
	err := row.Scan(&count)
	if err != nil {
		fmt.Println(err)
		return 0, err
	}

	if count != 0 {
		return 0, errors.InvalidOperationError{
			Operation: "Resolving vod appeal",
			Reason:    "Unresolved track appeals exist",
		}
	}

	now := time.Now().UTC().Round(time.Second)

	query = db.BuildQuery("UPDATE", models.VodAppealsTableName,
		"SET updated_at =", db.Param, ", resolved_at =", db.Param,
		"WHERE id = ", db.Param, "RETURNING vod_id")

	row = b.conn.QueryRow(ctx, "resolve_vod_appeal", query, now, now, appealID)

	var vodID int64
	err = row.Scan(&vodID)
	if err == db.ErrNoRows {
		return 0, errors.NotFoundError{Type: "Vod appeal", ID: int(appealID)}
	}

	vod, err := b.GetVodsByID(ctx, []int64{vodID}, edgemodels.VodFilterNone())
	if err != nil {
		return 0, err
	}

	if len(vod) == 0 {
		return 0, errors.NotFoundError{Type: "Vod", ID: int(vodID)}
	}

	dbVod := vod[0]

	// Fetch user vod properties
	p, err := b.GetUserVODProperties(ctx, dbVod.OwnerID)
	if err != nil {
		return 0, err
	}
	userVodProperties, err := p.AsVinylUserVODProperties()
	if err != nil {
		return 0, err
	}

	// If vods are stored forever, do nothing.
	if userVodProperties.SaveVODsForever {
		return vodID, nil
	}

	// Update the delete time for the vod if the deleted date earlier
	// than an updated time from now.
	query = db.BuildQuery("SELECT COUNT(1) FROM", models.AMRTableName,
		"WHERE unmuted_at is NULL", "AND vod_id = ", db.Param)
	row = b.conn.QueryRow(ctx, "vod_appeal_check_accept", query, dbVod.ID)

	var mutedCount int
	err = row.Scan(&mutedCount)
	if err != nil {
		return 0, err
	}

	vodStorageHours := userVodProperties.VODStorageDays * 24
	timeToDelete := dbVod.StartedOn.Add(time.Duration(vodStorageHours) * time.Hour)
	// Default extension time for rejected appeals
	additionalDays := time.Duration(72 * time.Hour)
	// Vod appeal was accepted since everything is unmuted
	if mutedCount == 0 {
		additionalDays = time.Duration(14 * 24 * time.Hour)
	}
	possibleDelete := time.Now().Add(additionalDays)

	if timeToDelete.Unix() < possibleDelete.Unix() {
		// Need to update the deletion time to extend it.
		updateFields := &edgemodels.VodUpdateInput{
			DeleteAt: edgemodels.NullInt64{Valid: true, Int64: possibleDelete.Unix()},
		}
		_, err = b.UpdateVod(ctx, vodID, updateFields)
		if err != nil {
			fmt.Println(err)
			return 0, err
		}
	}

	return vodID, nil
}
