package vinyldb

import (
	"regexp"
	"testing"
	"time"

	"code.justin.tv/vod/vinyl/datastore/vinyldb/models"
	edgemodels "code.justin.tv/vod/vinyl/models"

	sqlmock "github.com/DATA-DOG/go-sqlmock"
	. "github.com/smartystreets/goconvey/convey"

	"golang.org/x/net/context"
)

func TestResolveVodAppeal(t *testing.T) {
	Convey("ResolveVodAppeal", t, func() {
		mock, backend := Setup()
		Convey("returns an error if the unresolved track appeals exist", func() {
			appealID := int64(1238)

			query := `SELECT COUNT\(1\) FROM track_appeals` +
				` JOIN audible_magic_responses ON audible_magic_responses.id=track_appeals.audible_magic_response_id` +
				` JOIN vod_appeals ON vod_appeals.id=track_appeals.vod_appeal_id AND vod_appeals.vod_id=audible_magic_responses.vod_id` +
				` WHERE vod_appeal_id = \$1 AND audible_magic_responses.unmuted_at is NULL AND track_appeals.resolved_at is NULL`

			fakeResult := sqlmock.NewRows([]string{"count"})
			fakeResult.AddRow(2)
			mock.ExpectQuery(query).WithArgs(appealID).WillReturnRows(fakeResult)

			_, err := backend.ResolveVodAppeal(context.Background(), appealID)
			So(mock.ExpectationsWereMet(), ShouldBeNil)
			So(err.Error(), ShouldContainSubstring, "Resolving vod appeal")
			So(err.Error(), ShouldContainSubstring, "Unresolved track appeals exist")
		})

		Convey("returns an error if the vod appeal wasn't found", func() {
			appealID := int64(1238)

			countQuery := `SELECT COUNT\(1\) FROM track_appeals` +
				` JOIN audible_magic_responses ON audible_magic_responses.id=track_appeals.audible_magic_response_id` +
				` JOIN vod_appeals ON vod_appeals.id=track_appeals.vod_appeal_id AND vod_appeals.vod_id=audible_magic_responses.vod_id` +
				` WHERE vod_appeal_id = \$1 AND audible_magic_responses.unmuted_at is NULL AND track_appeals.resolved_at is NULL`
			fakeCountResult := sqlmock.NewRows([]string{"count"})
			fakeCountResult.AddRow(0)
			mock.ExpectQuery(countQuery).WithArgs(appealID).WillReturnRows(fakeCountResult)

			query := `UPDATE vod_appeals SET updated_at = \$1 , resolved_at = \$2 WHERE id = \$3 RETURNING vod_id`

			fakeResult := sqlmock.NewRows([]string{"vod_id"})
			mock.ExpectQuery(query).WithArgs(AnyTime{}, AnyTime{}, appealID).WillReturnRows(fakeResult)

			_, err := backend.ResolveVodAppeal(context.Background(), appealID)
			So(mock.ExpectationsWereMet(), ShouldBeNil)
			So(err.Error(), ShouldContainSubstring, "Vod appeal with ID 1238 not found")
		})

		Convey("returns an error if the track appeal was updated but the vod is not found", func() {
			appealID := int64(1238)

			countQuery := `SELECT COUNT\(1\) FROM track_appeals` +
				` JOIN audible_magic_responses ON audible_magic_responses.id=track_appeals.audible_magic_response_id` +
				` JOIN vod_appeals ON vod_appeals.id=track_appeals.vod_appeal_id AND vod_appeals.vod_id=audible_magic_responses.vod_id` +
				` WHERE vod_appeal_id = \$1 AND audible_magic_responses.unmuted_at is NULL AND track_appeals.resolved_at is NULL`
			fakeCountResult := sqlmock.NewRows([]string{"count"})
			fakeCountResult.AddRow(0)
			mock.ExpectQuery(countQuery).WithArgs(appealID).WillReturnRows(fakeCountResult)

			query := `UPDATE vod_appeals SET updated_at = \$1 , resolved_at = \$2 WHERE id = \$3 RETURNING vod_id`
			fakeResult := sqlmock.NewRows([]string{"vod_id"})
			fakeResult.AddRow(1)
			mock.ExpectQuery(query).WithArgs(AnyTime{}, AnyTime{}, appealID).WillReturnRows(fakeResult)

			vodQuery := regexp.QuoteMeta(models.FetchAllVODFieldsQuery() + ` WHERE id IN ($1)`)
			fakeVodResult := sqlmock.NewRows(models.VodFields)
			mock.ExpectQuery(vodQuery).WithArgs(1).WillReturnRows(fakeVodResult)

			_, err := backend.ResolveVodAppeal(context.Background(), appealID)
			So(mock.ExpectationsWereMet(), ShouldBeNil)
			So(err.Error(), ShouldContainSubstring, "Vod with ID 1 not found")
		})

		Convey("returns no error if the track appeal was updated and vods are stored forever", func() {
			appealID := int64(1238)

			countQuery := `SELECT COUNT\(1\) FROM track_appeals` +
				` JOIN audible_magic_responses ON audible_magic_responses.id=track_appeals.audible_magic_response_id` +
				` JOIN vod_appeals ON vod_appeals.id=track_appeals.vod_appeal_id AND vod_appeals.vod_id=audible_magic_responses.vod_id` +
				` WHERE vod_appeal_id = \$1 AND audible_magic_responses.unmuted_at is NULL AND track_appeals.resolved_at is NULL`
			fakeCountResult := sqlmock.NewRows([]string{"count"})
			fakeCountResult.AddRow(0)
			mock.ExpectQuery(countQuery).WithArgs(appealID).WillReturnRows(fakeCountResult)

			query := `UPDATE vod_appeals SET updated_at = \$1 , resolved_at = \$2 WHERE id = \$3 RETURNING vod_id`
			fakeResult := sqlmock.NewRows([]string{"vod_id"})
			fakeResult.AddRow(1)
			mock.ExpectQuery(query).WithArgs(AnyTime{}, AnyTime{}, appealID).WillReturnRows(fakeResult)

			vodQuery := regexp.QuoteMeta(models.FetchAllVODFieldsQuery() + ` WHERE id IN ($1)`)
			now := time.Now()
			fakeVodResult := sqlmock.NewRows(models.VodFields)
			ownerID := 111
			fakeVodResult.AddRow(
				1, nil, 34, "archive", nil, nil,
				now, nil, nil, nil, nil, nil, nil,
				1238, nil, nil, nil, nil, nil,
				0, ownerID, nil, nil, nil, nil, nil,
				now, edgemodels.StatusRecording, nil, nil, now,
				"hello", 0, nil, "", nil)

			mock.ExpectQuery(vodQuery).WithArgs(1).WillReturnRows(fakeVodResult)

			fakeVodThumbnails := sqlmock.NewRows([]string{"id", "vod_id", "path", "offset", "type"})
			mock.ExpectQuery(regexp.QuoteMeta(`SELECT id, vod_id, path, "offset", type FROM vod_thumbnails WHERE vod_id IN ( 1 )`)).WillReturnRows(fakeVodThumbnails)

			userQuery := models.FetchUserVODProperties() + ` WHERE user_id = \$1`
			fakeUserResult := sqlmock.NewRows(models.UserVODPropertiesFields)
			fakeUserResult.AddRow(ownerID, true, 209, nil, nil, nil, now, now)
			mock.ExpectQuery(userQuery).WithArgs(ownerID).WillReturnRows(fakeUserResult)

			vodID, err := backend.ResolveVodAppeal(context.Background(), appealID)
			So(mock.ExpectationsWereMet(), ShouldBeNil)
			So(err, ShouldBeNil)
			So(vodID, ShouldEqual, 1)
		})

		Convey("doesn't update the vod if the storage date is longer than the adjusted date", func() {
			appealID := int64(1238)

			countQuery := `SELECT COUNT\(1\) FROM track_appeals` +
				` JOIN audible_magic_responses ON audible_magic_responses.id=track_appeals.audible_magic_response_id` +
				` JOIN vod_appeals ON vod_appeals.id=track_appeals.vod_appeal_id AND vod_appeals.vod_id=audible_magic_responses.vod_id` +
				` WHERE vod_appeal_id = \$1 AND audible_magic_responses.unmuted_at is NULL AND track_appeals.resolved_at is NULL`
			fakeCountResult := sqlmock.NewRows([]string{"count"})
			fakeCountResult.AddRow(0)
			mock.ExpectQuery(countQuery).WithArgs(appealID).WillReturnRows(fakeCountResult)

			query := `UPDATE vod_appeals SET updated_at = \$1 , resolved_at = \$2 WHERE id = \$3 RETURNING vod_id`
			fakeResult := sqlmock.NewRows([]string{"vod_id"})
			fakeResult.AddRow(1)
			mock.ExpectQuery(query).WithArgs(AnyTime{}, AnyTime{}, appealID).WillReturnRows(fakeResult)

			vodQuery := regexp.QuoteMeta(models.FetchAllVODFieldsQuery() + ` WHERE id IN ($1)`)
			now := time.Now()
			fakeVodResult := sqlmock.NewRows(models.VodFields)
			ownerID := 111
			fakeVodResult.AddRow(
				1, nil, 34, "archive", nil, nil,
				now, nil, nil, nil, nil, nil, nil,
				1238, nil, nil, nil, nil, nil,
				0, ownerID, nil, nil, nil, nil, nil,
				now, edgemodels.StatusRecording, nil, nil, now,
				"hello", 0, nil, "", nil)

			mock.ExpectQuery(vodQuery).WithArgs(1).WillReturnRows(fakeVodResult)

			fakeVodThumbnails := sqlmock.NewRows([]string{"id", "vod_id", "path", "offset", "type"})
			mock.ExpectQuery(regexp.QuoteMeta(`SELECT id, vod_id, path, "offset", type FROM vod_thumbnails WHERE vod_id IN ( 1 )`)).WillReturnRows(fakeVodThumbnails)

			userQuery := models.FetchUserVODProperties() + ` WHERE user_id = \$1`
			fakeUserResult := sqlmock.NewRows(models.UserVODPropertiesFields)
			fakeUserResult.AddRow(ownerID, false, 100, nil, nil, nil, now, now)
			mock.ExpectQuery(userQuery).WithArgs(ownerID).WillReturnRows(fakeUserResult)

			// Storage length of 100 is longer than the adjusted date so no update is necessary.
			amrQuery := `SELECT COUNT\(1\) FROM audible_magic_responses WHERE unmuted_at is NULL AND vod_id = \$1`
			fakeAMRResult := sqlmock.NewRows([]string{"count"})
			fakeAMRResult.AddRow(0)
			mock.ExpectQuery(amrQuery).WithArgs(1).WillReturnRows(fakeAMRResult)

			vodID, err := backend.ResolveVodAppeal(context.Background(), appealID)
			So(mock.ExpectationsWereMet(), ShouldBeNil)
			So(err, ShouldBeNil)
			So(vodID, ShouldEqual, 1)
		})

		Convey("updates the vod if the storage date is longer than the adjusted date", func() {
			appealID := int64(1238)

			countQuery := `SELECT COUNT\(1\) FROM track_appeals` +
				` JOIN audible_magic_responses ON audible_magic_responses.id=track_appeals.audible_magic_response_id` +
				` JOIN vod_appeals ON vod_appeals.id=track_appeals.vod_appeal_id AND vod_appeals.vod_id=audible_magic_responses.vod_id` +
				` WHERE vod_appeal_id = \$1 AND audible_magic_responses.unmuted_at is NULL AND track_appeals.resolved_at is NULL`
			fakeCountResult := sqlmock.NewRows([]string{"count"})
			fakeCountResult.AddRow(0)
			mock.ExpectQuery(countQuery).WithArgs(appealID).WillReturnRows(fakeCountResult)

			query := `UPDATE vod_appeals SET updated_at = \$1 , resolved_at = \$2 WHERE id = \$3 RETURNING vod_id`
			fakeResult := sqlmock.NewRows([]string{"vod_id"})
			fakeResult.AddRow(1)
			mock.ExpectQuery(query).WithArgs(AnyTime{}, AnyTime{}, appealID).WillReturnRows(fakeResult)

			vodQuery := regexp.QuoteMeta(models.FetchAllVODFieldsQuery() + ` WHERE id IN ($1)`)
			now := time.Now()
			fakeVodResult := sqlmock.NewRows(models.VodFields)
			ownerID := 111
			fakeVodResult.AddRow(
				1, nil, 34, "archive", nil, nil,
				now, nil, nil, nil, nil, nil, nil,
				1238, nil, nil, nil, nil, nil,
				0, ownerID, nil, nil, nil, nil, nil,
				now, edgemodels.StatusRecording, nil, nil, now,
				"hello", 0, nil, "", nil)

			mock.ExpectQuery(vodQuery).WithArgs(1).WillReturnRows(fakeVodResult)

			fakeVodThumbnails := sqlmock.NewRows([]string{"id", "vod_id", "path", "offset", "type"})
			mock.ExpectQuery(regexp.QuoteMeta(`SELECT id, vod_id, path, "offset", type FROM vod_thumbnails WHERE vod_id IN ( 1 )`)).WillReturnRows(fakeVodThumbnails)

			userQuery := models.FetchUserVODProperties() + ` WHERE user_id = \$1`
			fakeUserResult := sqlmock.NewRows(models.UserVODPropertiesFields)
			fakeUserResult.AddRow(ownerID, false, 1, nil, nil, nil, now, now)
			mock.ExpectQuery(userQuery).WithArgs(ownerID).WillReturnRows(fakeUserResult)

			// Storage length of 1 is shorter than the adjusted date so update is necessary.
			amrQuery := `SELECT COUNT\(1\) FROM audible_magic_responses WHERE unmuted_at is NULL AND vod_id = \$1`
			fakeAMRResult := sqlmock.NewRows([]string{"count"})
			fakeAMRResult.AddRow(0)
			mock.ExpectQuery(amrQuery).WithArgs(1).WillReturnRows(fakeAMRResult)

			updateVodQuery := `UPDATE vods SET (.+) WHERE (.+) RETURNING (.+)`
			fakeUpdateResult := sqlmock.NewRows(models.VodFields)
			fakeUpdateResult.AddRow(1, nil, 34, "archive", nil, nil,
				now, nil, nil, nil, nil, nil, nil,
				1238, nil, nil, nil, nil, nil,
				0, 111, nil, nil, nil, nil, nil,
				now, edgemodels.StatusRecording, nil, nil, now,
				"happy", 0, nil, "", nil)

			mock.ExpectQuery(updateVodQuery).WithArgs(AnyEpoch{}, AnyTime{}, 1).WillReturnRows(fakeUpdateResult)

			fakeVodThumbnails = sqlmock.NewRows([]string{"id", "vod_id", "path", "offset", "type"})
			mock.ExpectQuery(regexp.QuoteMeta(`SELECT id, vod_id, path, "offset", type FROM vod_thumbnails WHERE vod_id IN ( 1 )`)).WillReturnRows(fakeVodThumbnails)

			vodID, err := backend.ResolveVodAppeal(context.Background(), appealID)
			So(mock.ExpectationsWereMet(), ShouldBeNil)
			So(err, ShouldBeNil)
			So(vodID, ShouldEqual, 1)
		})
	})
}
