package vinyldb

import (
	"database/sql"
	"errors"
	"testing"
	"time"

	"code.justin.tv/vod/vinyl/datastore/vinyldb/models"
	edgemodels "code.justin.tv/vod/vinyl/models"
	sqlmock "github.com/DATA-DOG/go-sqlmock"
	. "github.com/smartystreets/goconvey/convey"

	"golang.org/x/net/context"
)

func TestSetUserVideoPrivacyProperties(t *testing.T) {
	now := time.Now().UTC()
	Convey("SetUserVideoPrivacyProperties", t, func() {
		mock, backend := Setup()

		Convey("updates properties on an existing user", func() {
			userID := int64(1238)
			u := edgemodels.UserVideoPrivacyPropertiesInput{
				HideArchives: edgemodels.NullBool{Bool: true, Valid: true},
				UpdatedAt:    now,
				CreatedAt:    now,
			}

			query1 := `SELECT (.+) FROM ` + models.UserVideoPrivacyPropertiesTableName + ` WHERE user_id = (.+)`
			mock.ExpectQuery(query1).WithArgs(userID).WillReturnRows(sqlmock.NewRows([]string{"id"}).AddRow(1))

			query2 := `UPDATE user_video_privacy_properties SET (.+) WHERE (.+) RETURNING (.+)`
			fakeResult := sqlmock.NewRows(models.UserVideoPrivacyPropertiesFields).AddRow(userID, false, now, now)
			mock.ExpectQuery(query2).WithArgs(true, AnyTime{}, userID).WillReturnRows(fakeResult)

			res, err := backend.SetUserVideoPrivacyProperties(context.Background(), userID, &u)
			expected := &edgemodels.UserVideoPrivacyProperties{
				UserID:       userID,
				HideArchives: false,
				UpdatedAt:    now,
				CreatedAt:    now,
			}
			So(res, ShouldResemble, expected)
			So(mock.ExpectationsWereMet(), ShouldBeNil)
			So(err, ShouldBeNil)
		})

		// Delete this once we have our own DB which can upsert
		Convey("inserts new row for a new user", func() {
			userID := int64(1238)
			u := edgemodels.UserVideoPrivacyPropertiesInput{
				HideArchives: edgemodels.NullBool{Bool: true, Valid: true},
				UpdatedAt:    now,
				CreatedAt:    now,
			}

			query1 := `SELECT (.+) FROM ` + models.UserVideoPrivacyPropertiesTableName + ` WHERE user_id = (.+)`
			mock.ExpectQuery(query1).WithArgs(userID).WillReturnError(sql.ErrNoRows)

			query2 := `INSERT INTO user_video_privacy_properties (.+) VALUES (.+) RETURNING (.+)`
			fakeResult := sqlmock.NewRows(models.UserVideoPrivacyPropertiesFields).AddRow(userID, true, now, now)
			mock.ExpectQuery(query2).WithArgs(userID, true, AnyTime{}, AnyTime{}).WillReturnRows(fakeResult)

			res, err := backend.SetUserVideoPrivacyProperties(context.Background(), userID, &u)
			expected := &edgemodels.UserVideoPrivacyProperties{
				UserID:       userID,
				HideArchives: true,
				UpdatedAt:    now,
				CreatedAt:    now,
			}
			So(res, ShouldResemble, expected)
			So(mock.ExpectationsWereMet(), ShouldBeNil)
			So(err, ShouldBeNil)
		})

		Convey("neither insert nor update query run if select errors out", func() {
			userID := int64(1238)
			u := edgemodels.UserVideoPrivacyPropertiesInput{
				HideArchives: edgemodels.NullBool{Bool: true, Valid: true},
				UpdatedAt:    now,
				CreatedAt:    now,
			}

			query := `SELECT (.+) FROM ` + models.UserVideoPrivacyPropertiesTableName + ` WHERE user_id = (.+)`
			mock.ExpectQuery(query).WithArgs(userID).WillReturnError(errors.New("database is broken"))

			_, err := backend.SetUserVideoPrivacyProperties(context.Background(), userID, &u)
			So(mock.ExpectationsWereMet(), ShouldBeNil)
			So(err.Error(), ShouldContainSubstring, "database is broken")
		})
	})
}
