package vinyldb

import (
	"strconv"
	"strings"

	"code.justin.tv/chat/db"
	"code.justin.tv/vod/vinyl/datastore/vinyldb/models"
	"golang.org/x/net/context"
)

// SetViewcounts calls vinylDB to update vod counts as a batch.
func (b *Backend) SetViewcounts(ctx context.Context, counts map[int]int64) ([]*models.Vod, error) {
	query := db.BuildQuery(
		"UPDATE", models.VodTableName,
		"SET views_count = c.count",
		"FROM (VALUES", buildMappingTable(2, len(counts)), ") AS c (c_id, count)",
		"WHERE", models.VodTableName+".id = c.c_id",
		"RETURNING", strings.Join(models.VodFields, ","))
	rows, err := b.conn.Query(ctx, "set_viewcount", query, buildViewcountParams(counts)...)
	updatedVods, err := models.ReadVodRows(rows, err, b.logger)
	if err != nil {
		return nil, err
	}
	err = b.AttachVodThumbnails(ctx, updatedVods)
	if err != nil {
		return nil, err
	}

	return updatedVods, nil
}

func buildViewcountParams(counts map[int]int64) []interface{} {
	s := []interface{}{}
	for key, val := range counts {
		s = append(s, key, val)
	}
	return s
}

func buildMappingTable(perValue, total int) string {
	vals := []string{}
	for i := 0; i < total; i++ {
		base := i * perValue
		val := []string{}
		for j := 1; j <= perValue; j++ {
			val = append(val, "$"+strconv.Itoa(base+j)+"::integer")
		}
		vals = append(vals, "("+strings.Join(val, ",")+")")
	}

	return strings.Join(vals, ",")
}
