package vinyldb

import (
	"fmt"

	"code.justin.tv/chat/db"
	"code.justin.tv/vod/vinyl/datastore/vinyldb/models"
	"golang.org/x/net/context"
)

var pgPeriods = map[string]string{
	"month": "30 days",
	"week":  "7 days",
	"day":   "1 day",
}

// GetTopVods calls vinylDB to find the most popular vods from a game.
func (b *Backend) GetTopVods(ctx context.Context, broadcastType, language []string, game, period, sort string, limit, offset int) ([]*models.Vod, error) {
	//Temporary fix - if one of the broadcast_types is uploads and period is not specified - we use monthly period for uploads.
	uploadsPeriodSpecialCase := false
	nonUploadTypes := []string{}
	if period == "" {
		if len(broadcastType) > 1 {
			for _, bType := range broadcastType {
				if bType == "upload" {
					uploadsPeriodSpecialCase = true
				} else {
					nonUploadTypes = append(nonUploadTypes, bType)
				}
			}
			period = "week"
		}

		//If period is empty and only one broadcast type is specified - set period to month if upload, else week.
		if len(broadcastType) == 1 {
			if broadcastType[0] == "upload" {
				period = "month"
			} else {
				period = "week"
			}
		}
	}

	res := []*models.Vod{}

	params := []interface{}{}
	for _, b := range broadcastType {
		params = append(params, b)
	}
	queryParts := []interface{}{
		models.FetchAllVODFieldsQuery(),
		"WHERE broadcast_type IN ", db.Params(len(broadcastType)),
		models.Watchable(),
	}

	if len(language) > 0 {
		for _, l := range language {
			params = append(params, l)
		}
		queryParts = append(queryParts, "AND language IN ", db.Params(len(language)))
	}

	if game != "" {
		params = append(params, game)
		queryParts = append(queryParts, "AND game = ", db.Param)
	}

	if uploadsPeriodSpecialCase {
		for _, b := range nonUploadTypes {
			params = append(params, b)
		}
		weekInterval, _ := pgPeriods["week"]
		queryParts = append(queryParts, "AND ((broadcast_type IN ", db.Params(len(nonUploadTypes)), fmt.Sprintf("AND started_on > (now() - interval '%s')", weekInterval), ")")
		queryParts = append(queryParts, "OR ")
		params = append(params, "upload")
		monthInterval, _ := pgPeriods["month"]
		queryParts = append(queryParts, "(broadcast_type = ", db.Param, fmt.Sprintf("AND started_on > (now() - interval '%s')", monthInterval), "))")
	} else {
		if pgPeriod, ok := pgPeriods[period]; ok {
			queryParts = append(queryParts, fmt.Sprintf("AND started_on > (now() - interval '%s')", pgPeriod))
		}
	}

	params = append(params, limit, offset)
	queryParts = append(queryParts, pgSortQueryPart(sort), "LIMIT", db.Param, "OFFSET", db.Param)

	query := db.BuildQuery(queryParts...)

	rows, err := b.conn.Query(ctx, "get_top_vods", query, params...)
	res, err = models.ReadVodRows(rows, err, b.logger)
	if err != nil {
		return nil, err
	}
	err = b.AttachVodThumbnails(ctx, res)
	if err != nil {
		return nil, err
	}

	return res, err

}
