package vinyldb

import (
	"regexp"
	"testing"

	"code.justin.tv/vod/vinyl/datastore/vinyldb/models"

	. "github.com/smartystreets/goconvey/convey"
	"golang.org/x/net/context"
)

//should probably just mock out readvodrows
func TestGetTopVods(t *testing.T) {
	var broadcastType, language []string
	var game, period string
	var limit, offset int
	Convey("Top vods", t, func() {
		mock, reader := Setup()
		broadcastType = []string{"highlight"}
		language = []string{}
		game = ""
		period = "week"
		limit = 20
		offset = 0
		sort := "views"
		Convey("without a game param", func() {
			query := regexp.QuoteMeta(models.FetchAllVODFieldsQuery() + ` WHERE broadcast_type IN ($1)` +
				` AND deleted IS NOT TRUE AND viewable IS DISTINCT FROM 'private' AND status IN ('recording', 'recorded')` +
				` AND started_on > (now() - interval '7 days')` +
				` ORDER BY views_count DESC LIMIT $2 OFFSET $3`)
			mock.ExpectQuery(query).WithArgs(broadcastType[0], limit, offset)
			_, err := reader.GetTopVods(context.Background(), broadcastType, language, game, period, sort, limit, offset)
			So(mock.ExpectationsWereMet(), ShouldBeNil)
			So(err, ShouldNotBeNil)
		})

		Convey("with a game param", func() {
			game = "League of Legends"
			query := regexp.QuoteMeta(models.FetchAllVODFieldsQuery() + ` WHERE broadcast_type IN ($1)` +
				` AND deleted IS NOT TRUE AND viewable IS DISTINCT FROM 'private' AND status IN ('recording', 'recorded')` +
				` AND game = $2` +
				` AND started_on > (now() - interval '7 days')` +
				` ORDER BY views_count DESC LIMIT $3 OFFSET $4`)
			mock.ExpectQuery(query).WithArgs(broadcastType[0], game, limit, offset)

			_, err := reader.GetTopVods(context.Background(), broadcastType, language, game, period, sort, limit, offset)
			So(mock.ExpectationsWereMet(), ShouldBeNil)
			So(err, ShouldNotBeNil)
		})

		Convey("with a language param", func() {
			language = []string{"en"}
			game = "League of Legends"
			query := regexp.QuoteMeta(models.FetchAllVODFieldsQuery() + ` WHERE broadcast_type IN ($1)` +
				` AND deleted IS NOT TRUE AND viewable IS DISTINCT FROM 'private' AND status IN ('recording', 'recorded')` +
				` AND language IN ($2)` +
				` AND game = $3` +
				` AND started_on > (now() - interval '7 days')` +
				` ORDER BY views_count DESC LIMIT $4 OFFSET $5`)
			mock.ExpectQuery(query).WithArgs(broadcastType[0], language[0], game, limit, offset)

			_, err := reader.GetTopVods(context.Background(), broadcastType, language, game, period, sort, limit, offset)
			So(mock.ExpectationsWereMet(), ShouldBeNil)
			So(err, ShouldNotBeNil)
		})

		Convey("period", func() {
			Convey("empty period without upload type", func() {
				period = ""
				query := regexp.QuoteMeta(models.FetchAllVODFieldsQuery() + ` WHERE broadcast_type IN ($1)` +
					` AND deleted IS NOT TRUE AND viewable IS DISTINCT FROM 'private' AND status IN ('recording', 'recorded')` +
					` AND started_on > (now() - interval '7 days')` +
					` ORDER BY views_count DESC LIMIT $2 OFFSET $3`)
				mock.ExpectQuery(query).WithArgs(broadcastType[0], limit, offset)

				_, err := reader.GetTopVods(context.Background(), broadcastType, language, game, period, sort, limit, offset)
				So(mock.ExpectationsWereMet(), ShouldBeNil)
				So(err, ShouldNotBeNil)
			})
			Convey("day", func() {
				period = "day"
				query := regexp.QuoteMeta(models.FetchAllVODFieldsQuery() + ` WHERE broadcast_type IN ($1)` +
					` AND deleted IS NOT TRUE AND viewable IS DISTINCT FROM 'private' AND status IN ('recording', 'recorded')` +
					` AND started_on > (now() - interval '1 day')` +
					` ORDER BY views_count DESC LIMIT $2 OFFSET $3`)
				mock.ExpectQuery(query).WithArgs(broadcastType[0], limit, offset)

				_, err := reader.GetTopVods(context.Background(), broadcastType, language, game, period, sort, limit, offset)
				So(mock.ExpectationsWereMet(), ShouldBeNil)
				So(err, ShouldNotBeNil)
			})
			Convey("week", func() {
				period = "week"
				query := regexp.QuoteMeta(models.FetchAllVODFieldsQuery() + ` WHERE broadcast_type IN ($1)` +
					` AND deleted IS NOT TRUE AND viewable IS DISTINCT FROM 'private' AND status IN ('recording', 'recorded')` +
					` AND started_on > (now() - interval '7 days')` +
					` ORDER BY views_count DESC LIMIT $2 OFFSET $3`)
				mock.ExpectQuery(query).WithArgs(broadcastType[0], limit, offset)

				_, err := reader.GetTopVods(context.Background(), broadcastType, language, game, period, sort, limit, offset)
				So(mock.ExpectationsWereMet(), ShouldBeNil)
				So(err, ShouldNotBeNil)
			})
			Convey("month", func() {
				period = "month"
				query := regexp.QuoteMeta(models.FetchAllVODFieldsQuery() + ` WHERE broadcast_type IN ($1)` +
					` AND deleted IS NOT TRUE AND viewable IS DISTINCT FROM 'private' AND status IN ('recording', 'recorded')` +
					` AND started_on > (now() - interval '30 days')` +
					` ORDER BY views_count DESC LIMIT $2 OFFSET $3`)
				mock.ExpectQuery(query).WithArgs(broadcastType[0], limit, offset)

				_, err := reader.GetTopVods(context.Background(), broadcastType, language, game, period, sort, limit, offset)
				So(mock.ExpectationsWereMet(), ShouldBeNil)
				So(err, ShouldNotBeNil)
			})
			Convey("all", func() {
				period = "all"
				query := regexp.QuoteMeta(models.FetchAllVODFieldsQuery() + ` WHERE broadcast_type IN ($1)` +
					` AND deleted IS NOT TRUE AND viewable IS DISTINCT FROM 'private' AND status IN ('recording', 'recorded')` +
					` ORDER BY views_count DESC LIMIT $2 OFFSET $3`)
				mock.ExpectQuery(query).WithArgs(broadcastType[0], limit, offset)

				_, err := reader.GetTopVods(context.Background(), broadcastType, language, game, period, sort, limit, offset)
				So(mock.ExpectationsWereMet(), ShouldBeNil)
				So(err, ShouldNotBeNil)
			})
		})

		Convey("multiple broadcast types", func() {
			broadcastType = []string{"archive", "upload", "highlight"}
			query := regexp.QuoteMeta(models.FetchAllVODFieldsQuery() + ` WHERE broadcast_type IN ($1, $2, $3)` +
				` AND deleted IS NOT TRUE AND viewable IS DISTINCT FROM 'private' AND status IN ('recording', 'recorded')` +
				` AND started_on > (now() - interval '7 days')` +
				` ORDER BY views_count DESC LIMIT $4 OFFSET $5`)
			mock.ExpectQuery(query).WithArgs(broadcastType[0], broadcastType[1], broadcastType[2], limit, offset)

			_, err := reader.GetTopVods(context.Background(), broadcastType, language, game, period, sort, limit, offset)
			So(mock.ExpectationsWereMet(), ShouldBeNil)
			So(err, ShouldNotBeNil)
			Convey("with game", func() {
				game = "Hearthstone"
				query := regexp.QuoteMeta(models.FetchAllVODFieldsQuery() + ` WHERE broadcast_type IN ($1, $2, $3)` +
					` AND deleted IS NOT TRUE AND viewable IS DISTINCT FROM 'private' AND status IN ('recording', 'recorded')` +
					` AND game = $4` +
					` AND started_on > (now() - interval '7 days')` +
					` ORDER BY views_count DESC LIMIT $5 OFFSET $6`)
				mock.ExpectQuery(query).WithArgs(broadcastType[0], broadcastType[1], broadcastType[2], game, limit, offset)

				_, err := reader.GetTopVods(context.Background(), broadcastType, language, game, period, sort, limit, offset)
				So(mock.ExpectationsWereMet(), ShouldBeNil)
				So(err, ShouldNotBeNil)
			})
		})

		Convey("multiple broadcast types including upload - with empty period", func() {
			broadcastType = []string{"archive", "upload", "highlight"}
			query := regexp.QuoteMeta(models.FetchAllVODFieldsQuery() + ` WHERE broadcast_type IN ($1, $2, $3)` +
				` AND deleted IS NOT TRUE AND viewable IS DISTINCT FROM 'private' AND status IN ('recording', 'recorded')` +
				` AND ((broadcast_type IN ($4, $5) AND started_on > (now() - interval '7 days') ) OR` +
				` (broadcast_type = $6 AND started_on > (now() - interval '30 days') ))` +
				` ORDER BY views_count DESC LIMIT $7 OFFSET $8`)
			mock.ExpectQuery(query).WithArgs(broadcastType[0], broadcastType[1], broadcastType[2], broadcastType[0], broadcastType[2], broadcastType[1], limit, offset)

			period = ""
			_, err := reader.GetTopVods(context.Background(), broadcastType, language, game, period, sort, limit, offset)
			So(mock.ExpectationsWereMet(), ShouldBeNil)
			So(err, ShouldNotBeNil)
		})

		Convey("multiple broadcast types not including upload - with empty period", func() {
			broadcastType = []string{"archive", "highlight"}
			query := regexp.QuoteMeta(models.FetchAllVODFieldsQuery() + ` WHERE broadcast_type IN ($1, $2)` +
				` AND deleted IS NOT TRUE AND viewable IS DISTINCT FROM 'private' AND status IN ('recording', 'recorded')` +
				` AND started_on > (now() - interval '7 days')` +
				` ORDER BY views_count DESC LIMIT $3 OFFSET $4`)
			mock.ExpectQuery(query).WithArgs(broadcastType[0], broadcastType[1], limit, offset)

			period = ""
			_, err := reader.GetTopVods(context.Background(), broadcastType, language, game, period, sort, limit, offset)
			So(mock.ExpectationsWereMet(), ShouldBeNil)
			So(err, ShouldNotBeNil)
		})

		Convey("upload broadcast type - with empty period", func() {
			broadcastType = []string{"upload"}
			query := regexp.QuoteMeta(models.FetchAllVODFieldsQuery() + ` WHERE broadcast_type IN ($1)` +
				` AND deleted IS NOT TRUE AND viewable IS DISTINCT FROM 'private' AND status IN ('recording', 'recorded')` +
				` AND started_on > (now() - interval '30 days')` +
				` ORDER BY views_count DESC LIMIT $2 OFFSET $3`)
			mock.ExpectQuery(query).WithArgs(broadcastType[0], limit, offset)

			period = ""
			_, err := reader.GetTopVods(context.Background(), broadcastType, language, game, period, sort, limit, offset)
			So(mock.ExpectationsWereMet(), ShouldBeNil)
			So(err, ShouldNotBeNil)
		})

	})
}
