package main

import (
	"database/sql"
	"flag"
	"fmt"
	"time"

	"code.justin.tv/vod/vinyl/datastore/vinyldb/utils"
	"code.justin.tv/vod/vinyl/markdown"

	_ "github.com/lib/pq"
)

const sleepTime = time.Millisecond * 150
const readBatchSize = 100

func closeDB(conn *sql.DB) {
	err := conn.Close()
	if err != nil {
		fmt.Println("Had trouble closing connection to DB ", err)
	}
}

type nullInt struct {
	Valid bool
	Int   int64
}

func batchBackfill(conn *sql.DB, batchSize int, idLimit nullInt) (nullInt, int, error) {
	// In order to speed up the read query as time goes on, we will limit the ID to be less
	// than the smallest ID from the previous batch of vods. This works because
	// we're ordering by ID DESC anyway.
	totalRows := 0

	query := fmt.Sprintf("SELECT id, description from vods WHERE description is not null AND description_html is null ORDER BY id DESC LIMIT %d", batchSize)
	if idLimit.Valid {
		query = fmt.Sprintf("SELECT id, description from vods WHERE description is not null AND description_html is null AND id < %d ORDER BY id DESC LIMIT %d", idLimit.Int, batchSize)
	}
	fmt.Println(query)
	rows, err := conn.Query(query)
	if err != nil {
		fmt.Println("error fetching vod rows from db: ", err)
		return nullInt{Valid: false}, totalRows, err
	}
	defer utils.CloseRows(rows, nil)
	lowestID := nullInt{Valid: false}

	for rows.Next() {
		totalRows++
		var description sql.NullString
		var vodID int64
		if err := rows.Scan(&vodID, &description); err != nil {
			fmt.Println("error scanning vod row: ", err)
			continue
		}
		if !lowestID.Valid {
			lowestID.Valid = true
			lowestID.Int = vodID
		} else if lowestID.Valid && vodID < lowestID.Int {
			lowestID.Int = vodID
		}
		if description.Valid {
			renderedMarkdown := markdown.ConvertMarkdown(description.String)
			err := saveHTMLToDB(conn, renderedMarkdown, vodID)
			if err != nil {
				return nullInt{Valid: false}, totalRows, err
			}
		}
	}

	return lowestID, totalRows, nil
}

func saveHTMLToDB(conn *sql.DB, html string, vodID int64) error {
	// Since fetching the number of rows to update and the UPDATE query happen in separate transactions,
	// there's a race condition between this backfill script and a user manually updating the description.
	// Therefore, we only want to update the vod if the description_html column is still null
	_, err := conn.Exec("UPDATE vods SET description_html = $1 where id = $2 AND description_html is null", html, vodID)
	if err != nil {
		fmt.Printf("error saving vod %d with id html to DB: %v\n", vodID, err)
		return err
	}

	fmt.Println("Updated vod with id: ", vodID)
	return nil
}

func main() {
	hostPtr := flag.String("host", "vinyl-staging-master.cifgffw7w2ar.us-west-2.rds.amazonaws.com", "db hostname")
	dbPtr := flag.String("db", "vinyl", "db name")
	userPtr := flag.String("user", "vinyl", "user name")
	passwordPtr := flag.String("password", "", "password")
	portPtr := flag.Int("port", 5432, "port")
	flag.Parse()

	conn, err := sql.Open("postgres", fmt.Sprintf("host=%s port=%d dbname=%s user=%s password=%s", *hostPtr, *portPtr, *dbPtr, *userPtr, *passwordPtr))
	if err != nil {
		fmt.Println("error opening connection to postgres: ", err)
		return
	}
	defer closeDB(conn)

	idLimit := nullInt{Valid: false}
	rowsRead := 0
	for {
		idLimit, rowsRead, err = batchBackfill(conn, readBatchSize, idLimit)
		if err != nil {
			break
		}
		if rowsRead == 0 {
			fmt.Println("Finished backfilling vods!")
			break
		}
		time.Sleep(sleepTime)
	}
}
