package errors

import (
	"fmt"
	"net/http"
	"strings"
	"time"
)

// MissingParamError is returned when a required parameter is missing.
type MissingParamError struct {
	ParamName string
}

// Error returns the string representation of a MissingParamError.
func (e MissingParamError) Error() string {
	return "Missing required param: " + e.ParamName
}

// DisallowedParamError is returned when an invalid parameter name is passed in,
// usually in an udpate operation.
type DisallowedParamError struct {
	ParamName string
}

// Error returns the string representation of a DisallowedParamError.
func (e DisallowedParamError) Error() string {
	return "Parameter not allowed: " + e.ParamName
}

// IntegerParseError is returned when a parameter that is supposed to be an integer is not parsable.
type IntegerParseError struct {
	ParamName  string
	ParamValue string
}

// Error returns the string representation of a IntegerParseError.
func (e IntegerParseError) Error() string {
	return e.ParamName + " is not a number: " + e.ParamValue
}

// TimeParseError is returned when a time parameter is not in a recognizable form
type TimeParseError struct {
	ParamName  string
	ParamValue string
}

func (e TimeParseError) Error() string {
	return e.ParamName + " is not a valid time: " + e.ParamValue
}

// RequestBodyError is returned when the body of a request cannot be read or contains
// the wrong type of field.
type RequestBodyError struct {
	ErrorString string
}

// Error returns the string representation of a RequestBodyError.
func (e RequestBodyError) Error() string {
	return e.ErrorString
}

// InvalidValueError is returned when a parameter is passed in with a value that is not
// part of a validated list.
type InvalidValueError struct {
	ParamName string
	Value     interface{}
}

// Error returns the string representation of a InvalidValueError.
func (e InvalidValueError) Error() string {
	return fmt.Sprintf("Invalid value for %s: %v", e.ParamName, e.Value)
}

// InvalidTimeInterval is returned when two time parameters don't form a valid interval
type InvalidTimeInterval struct {
	StartTimeName  string
	StartTimeValue time.Time
	EndTimeName    string
	EndTimeValue   time.Time
}

// Error returns the string representation of the InvalidTimeInterval and the values given.
func (e InvalidTimeInterval) Error() string {
	return fmt.Sprintf("%s must be before %s. Given %v and %v.", e.StartTimeName, e.EndTimeName, e.StartTimeValue, e.EndTimeValue)
}

// TitleTooShortError is returned when a corresponding vod's title is too short.
type TitleTooShortError struct {
}

// Error returns the string representation of a TitleTooShortError.
func (e TitleTooShortError) Error() string {
	return "Title cannot be empty"
}

// TitleTooLongError is returned when a corresponding vod's title is too long.
type TitleTooLongError struct {
}

// Error returns the string representation of a TitleTooLongError.
func (e TitleTooLongError) Error() string {
	return "Title must be less than or equal to 100 characters"
}

// PerTagTooLongError is returned when a tag in the list is too long.
type PerTagTooLongError struct {
}

// Error returns the string representation of a PerTagTooLongError.
func (e PerTagTooLongError) Error() string {
	return "One or more of your tags are too long."
}

// TagFieldTooLongError is returned when a vod's tags have too many characters.
type TagFieldTooLongError struct {
}

// Error returns the string representation of a TagFieldTooLongError.
func (e TagFieldTooLongError) Error() string {
	return "Your tags are too long."
}

// NotFoundError is returned when a corresponding object does not exist.
type NotFoundError struct {
	Type string
	ID   int // TODO: this should be int64
}

// Error returns the string representation of a NotFoundError.
func (e NotFoundError) Error() string {
	return fmt.Sprintf("%s with ID %d not found", e.Type, e.ID)
}

// TooManyUploadsError is return when a start and end time yield a negative duration.
type TooManyUploadsError struct {
}

// Error returns the string representation of an TooManyUploadsError.
func (e TooManyUploadsError) Error() string {
	return fmt.Sprintf("You have too many uploads. Please wait for existing uploads to finish.")
}

// InvalidDurationError is return when a start and end time yield a negative duration.
type InvalidDurationError struct {
	StartSeconds int64
	EndSeconds   int64
}

// Error returns the string representation of an InvalidDurationError.
func (e InvalidDurationError) Error() string {
	return fmt.Sprintf("start seconds %d with end seconds %d yields non-positive duration %d", e.StartSeconds, e.EndSeconds, (e.EndSeconds - e.StartSeconds))
}

// InvalidOperationError is returned when an operation is not allowed.
type InvalidOperationError struct {
	Operation string
	Reason    string
}

// Error returns the string representation of an InvalidOperationError.
func (e InvalidOperationError) Error() string {
	return fmt.Sprintf("%s is not allowed: %s", e.Operation, e.Reason)
}

// AudreyClientError is returned when the Audrey client serves an error.
type AudreyClientError struct {
	Reason string
}

// Error returns the error from the Audrey client
func (e AudreyClientError) Error() string {
	return fmt.Sprintf("Audrey client: %s", e.Reason)
}

// MatchError takes in an error, and matches it to the appropriate status response code.
func MatchError(e interface{}) int {
	switch e.(type) {
	case MissingParamError, IntegerParseError, RequestBodyError, InvalidValueError,
		DisallowedParamError, TimeParseError, InvalidTimeInterval, InvalidDurationError,
		TitleTooShortError, TitleTooLongError, TooManyUploadsError, PerTagTooLongError,
		TagFieldTooLongError, InvalidStateTransitionError:
		return http.StatusBadRequest
	case InvalidOperationError:
		return http.StatusForbidden
	case NotFoundError:
		return http.StatusNotFound
	case UnauthorizedError:
		return http.StatusUnauthorized
	case AudreyClientError:
		reason := e.(AudreyClientError).Reason
		if strings.Contains(reason, "429") {
			return http.StatusTooManyRequests
		} else if strings.Contains(reason, "400") {
			return http.StatusBadRequest
		}
		return http.StatusInternalServerError
	default:
		return http.StatusInternalServerError
	}
}

// NotificationTypeMissingError is returned when a notification's type is missing/empty.
type NotificationTypeMissingError struct {
}

// Error returns the string representation of a NotificationTypeMissingError.
func (e NotificationTypeMissingError) Error() string {
	return "Notification Type cannot be empty"
}

// NotificationTypeInvalidError is returned when a notification's type is invalid.
type NotificationTypeInvalidError struct {
	Type string
}

// Error returns the string representation of a NotificationTypeInvalidError.
func (e NotificationTypeInvalidError) Error() string {
	return fmt.Sprintf("Notification Type is not valid: %s", e.Type)
}

// YoutubeExportFeatureDisabledError is returned when the feature is disabled.
type YoutubeExportFeatureDisabledError struct {
}

// Error returns the string representation of a YoutubeExportFeatureDisabledError.
func (e YoutubeExportFeatureDisabledError) Error() string {
	return "This feature is temporarily disabled. Please try again later."
}

// UnauthorizedError is returned when a user is not authorized to perform an action
type UnauthorizedError struct {
}

// Error returns the string representation of a NotificationTypeInvalidError.
func (e UnauthorizedError) Error() string {
	return "You are not authorized"
}

// InvalidStateTransitionError is returned when a requester attempts to transition VOD status into an invalid state
type InvalidStateTransitionError struct {
	SourceState string
	TargetState string
}

// Error returns the string representation of a InvalidStateTransitionError.
func (e InvalidStateTransitionError) Error() string {
	return fmt.Sprintf("Invalid state transition: %s -> %s", e.SourceState, e.TargetState)
}
