package main

import (
	"errors"
	"fmt"
	"log"
	"os"
	"path"
	"strconv"

	"code.justin.tv/common/config"
	"code.justin.tv/common/twitchhttp"
	connections "code.justin.tv/identity/connections/client"
	"code.justin.tv/vod/vinyl/api"
	"code.justin.tv/vod/vinyl/auth"
	"code.justin.tv/vod/vinyl/backend"
	"code.justin.tv/vod/vinyl/clients"
	"code.justin.tv/vod/vinyl/datastore/vinyldb"
	appSettings "code.justin.tv/web/app-settings/client"
	"github.com/joho/godotenv"
)

var (
	rdsMasterHostConfKey = "rds-Master-HostName"
	rdsMasterPortConfKey = "rds-Master-Port"
	rdsSlaveHostConfKey  = "rds-Slave-HostName"
	rdsSlavePortConfKey  = "rds-Slave-Port"
	vinylDBNameConfKey   = "rds-db-name"

	kinesisStream      = "kinesis-stream"
	kinesisBatchStream = "kinesis-batch-stream"

	appSettingsHost = "app-settings-host"
	connectionsHost = "connections-host"
)

func init() {
	config.Register(map[string]string{
		rdsMasterHostConfKey: "vinyl-staging-master.cifgffw7w2ar.us-west-2.rds.amazonaws.com",
		rdsMasterPortConfKey: "5432",
		rdsSlaveHostConfKey:  "vinyl-staging-slave-001a.cifgffw7w2ar.us-west-2.rds.amazonaws.com",
		rdsSlavePortConfKey:  "5432",
		vinylDBNameConfKey:   "vinyl",

		kinesisStream:      "searchindexer-staging",
		kinesisBatchStream: "searchindexer-staging-batch",

		appSettingsHost: "app-settings.prod.us-west2.justin.tv",
		connectionsHost: "connections.internal.twitch.tv",
	})
}

func main() {
	env := os.Getenv("ENVIRONMENT")
	if env == "" {
		env = "development"
	}

	// Set environments variables based on config file in configuration/environments
	// They will override command-line flags.
	configFile := path.Join("configuration", "environments", env)
	err := godotenv.Load(configFile)
	if err != nil {
		log.Fatal("Error loading environment config file: ", configFile)
	}

	// To add app-specific config flags, you can call config.Register()
	// https://godoc.internal.justin.tv/code.justin.tv/common/config
	err = config.Parse()
	if err != nil {
		log.Fatal(err)
	}

	rdsSlaveHost := config.Resolve(rdsSlaveHostConfKey)
	rdsSlavePort, err := strconv.Atoi(config.Resolve(rdsSlavePortConfKey))
	if err != nil {
		log.Fatal(errors.New("Non-integer port:" + config.Resolve(rdsSlavePortConfKey)))
	}
	rdsMasterHost := config.Resolve(rdsMasterHostConfKey)
	rdsMasterPort, err := strconv.Atoi(config.Resolve(rdsMasterPortConfKey))
	if err != nil {
		log.Fatal(errors.New("Non-integer port:" + config.Resolve(rdsMasterPortConfKey)))
	}
	vinylDBName := config.Resolve(vinylDBNameConfKey)

	rdsReaderConf := clients.NewRdsReaderConf(rdsSlaveHost, rdsSlavePort, vinylDBName)
	rdsMasterReaderConf := clients.NewRdsMasterReaderConf(rdsMasterHost, rdsMasterPort, vinylDBName)
	rdsWriterConf := clients.NewRdsWriterConf(rdsMasterHost, rdsMasterPort, vinylDBName)

	rdsReaderConn, err := clients.NewPostgresConn(clients.VinylDBUser, clients.VinylDBPassword, rdsReaderConf)
	if err != nil {
		log.Fatal(err)
	}
	fmt.Printf("Connected to Postgres reader at %s:%d with user %s\n", rdsSlaveHost, rdsSlavePort, clients.VinylDBUser)
	rdsMasterReaderConn, err := clients.NewPostgresConn(clients.VinylDBUser, clients.VinylDBPassword, rdsMasterReaderConf)
	if err != nil {
		log.Fatal(err)
	}
	fmt.Printf("Connected to Postgres master reader at %s:%d with user %s\n", rdsMasterHost, rdsMasterPort, clients.VinylDBUser)
	rdsWriterConn, err := clients.NewPostgresConn(clients.VinylDBUser, clients.VinylDBPassword, rdsWriterConf)
	if err != nil {
		log.Fatal(err)
	}
	fmt.Printf("Connected to Postgres master writer at %s:%d with user %s\n", rdsMasterHost, rdsMasterPort, clients.VinylDBUser)

	reader := vinyldb.New(rdsReaderConn, "rds_reader", config.Statsd(), config.RollbarErrorLogger())
	masterReader := vinyldb.New(rdsMasterReaderConn, "rds_master_reader", config.Statsd(), config.RollbarErrorLogger())
	writer := vinyldb.New(rdsWriterConn, "rds_writer", config.Statsd(), config.RollbarErrorLogger())

	partnershipsClient, err := clients.NewPartnershipsClient(config.Statsd())
	if err != nil {
		log.Fatal(err)
	}

	railsClient, err := clients.NewRailsClient(config.Statsd())
	if err != nil {
		log.Fatal(err)
	}

	searchIndexerClient := clients.NewSearchIndexer(config.Resolve(kinesisStream), config.Resolve(kinesisBatchStream), config.Statsd())

	usersClient, err := clients.NewUsersServiceClient(config.Statsd())
	if err != nil {
		log.Fatal(err)
	}

	audreyClient, err := clients.NewAudreyClient(config.Statsd())
	if err != nil {
		log.Fatal(err)
	}

	pushyClient, err := clients.NewPushyClient(config.Statsd())
	if err != nil {
		log.Fatal(err)
	}

	appSettingsClient, err := appSettings.NewClient(twitchhttp.ClientConf{
		Host: config.MustResolve("app-settings-host"),
	})
	if err != nil {
		log.Fatal(err)
	}

	connectionsClient, err := connections.NewClient(twitchhttp.ClientConf{
		Host: config.MustResolve("connections-host"),
	})
	if err != nil {
		log.Fatal(err)
	}

	spadeClient, err := clients.NewSpadeClient()

	hostname, err := os.Hostname()
	if err != nil {
		log.Println(err)
	}
	searchIndexerLogStream := "searchindexer-" + hostname
	searchIndexerLog, err := clients.NewCloudwatchLogStream("vinyl-"+config.Environment(), searchIndexerLogStream)
	if err != nil {
		log.Println(err)
	}

	backend := &backend.Backend{
		Reader:             reader,
		MasterReader:       masterReader,
		Writer:             writer,
		Stats:              config.Statsd(),
		ErrorLogger:        config.RollbarErrorLogger(),
		SearchIndexerLog:   searchIndexerLog,
		AppSettingsClient:  appSettingsClient,
		AudreyClient:       audreyClient,
		ConnectionsClient:  connectionsClient,
		PartnershipsClient: partnershipsClient,
		PushyClient:        pushyClient,
		RailsClient:        railsClient,
		SpadeClient:        spadeClient,
		SearchIndexer:      searchIndexerClient,
		UsersClient:        usersClient,
		Usher:              clients.NewUsherClient(),
	}

	authHandler := auth.NewHandler("auth/key.pub")

	server, err := api.NewServer(config.Statsd(), config.RollbarErrorLogger(), backend, authHandler)
	if err != nil {
		log.Fatal(err)
	}

	log.Fatal(twitchhttp.ListenAndServe(server))
}
