package markdown

import (
	"bytes"
	"fmt"
	"log"
	"regexp"

	"github.com/microcosm-cc/bluemonday"
	"github.com/russross/blackfriday"
)

type markdown struct {
	renderer   *renderer
	policy     *bluemonday.Policy
	extensions int
}

var markdownInstance *markdown

type renderer struct {
	*blackfriday.Html
}

func init() {
	htmlExtensions := 0
	htmlExtensions |= blackfriday.HTML_SKIP_HTML
	htmlExtensions |= blackfriday.HTML_SKIP_IMAGES
	htmlExtensions |= blackfriday.HTML_HREF_TARGET_BLANK
	htmlExtensions |= blackfriday.HTML_NOFOLLOW_LINKS
	htmlExtensions |= blackfriday.HTML_NOREFERRER_LINKS
	htmlExtensions |= blackfriday.HTML_SAFELINK

	rendererInstance := &renderer{Html: blackfriday.HtmlRenderer(htmlExtensions, "", "").(*blackfriday.Html)}
	extensions := 0
	extensions |= blackfriday.EXTENSION_AUTOLINK
	extensions |= blackfriday.EXTENSION_HARD_LINE_BREAK

	policyInstance := bluemonday.UGCPolicy()
	policyInstance.AllowAttrs("rel").OnElements("a")
	policyInstance.AllowAttrs("target").OnElements("a")

	markdownInstance = &markdown{rendererInstance, policyInstance, extensions}
}

// ConvertMarkdown converts markdown into HTML and sanitize HTML output
func ConvertMarkdown(markdown string) string {
	input := []byte(markdown)
	unsafe := blackfriday.Markdown(input, markdownInstance.renderer, markdownInstance.extensions)
	html := string(markdownInstance.policy.SanitizeBytes(unsafe))
	// Hack to get rid of extra break from rendering last paragraph
	r, err := regexp.Compile("<br><br>$")
	if err != nil {
		log.Println(err)
	} else {
		html = r.ReplaceAllLiteralString(html, "<br>")
	}
	return html
}

func (options *renderer) Paragraph(out *bytes.Buffer, text func() bool) {
	text()
	_, err := out.WriteString("<br><br>")
	if err != nil {
		log.Println(err)
	}
}

func (options *renderer) Header(out *bytes.Buffer, text func() bool, level int, id string) {
	marker := out.Len()
	doubleSpace(out)
	level = max(level, 3)

	_, err := out.WriteString(fmt.Sprintf("<h%d>", level))
	if err != nil {
		log.Println(err)
	}

	if !text() {
		out.Truncate(marker)
		return
	}

	_, err = out.WriteString(fmt.Sprintf("</h%d>\n", level))
	if err != nil {
		log.Println(err)
	}
}

func doubleSpace(out *bytes.Buffer) {
	if out.Len() > 0 {
		err := out.WriteByte('\n')
		if err != nil {
			log.Println(err)
		}
	}
}

func max(a, b int) int {
	if a > b {
		return a
	}
	return b
}
