package markdown

import (
	"testing"

	. "github.com/smartystreets/goconvey/convey"
)

func TestConvertMarkdown(t *testing.T) {
	Convey("Markdown", t, func() {
		Convey("should filter xss attacks via links", func() {
			html := ConvertMarkdown(`[test](javascript:alert('xss'))`)
			So(html, ShouldNotContainSubstring, "javascript")
		})

		Convey("should filter inline html", func() {
			html := ConvertMarkdown(`<html><a>test</a></html>`)
			So(html, ShouldNotContainSubstring, "<html>")
			So(html, ShouldNotContainSubstring, "<a>")
		})

		Convey("should create links with nofollow, noreferrer, noopener, and target=_blank", func() {
			html := ConvertMarkdown(`[foo](https://google.com)`)
			So(html, ShouldContainSubstring, `<a href="https://google.com" rel="nofollow noreferrer noopener" target="_blank">foo</a>`)
		})

		Convey("should not allow inline embeds for links", func() {
			html := ConvertMarkdown(`![foo](https://google.com)`)
			So(html, ShouldNotContainSubstring, "google.com")
		})

		Convey("should allow auto-linking", func() {
			html := ConvertMarkdown(`https://www.google.com`)
			So(html, ShouldContainSubstring, `<a href="https://www.google.com" rel="nofollow noreferrer noopener" target="_blank">https://www.google.com</a>`)
		})

		Convey("should allow styling text", func() {
			html := ConvertMarkdown(`**bold**`)
			So(html, ShouldContainSubstring, `<strong>bold</strong>`)
			html = ConvertMarkdown(`*italicized*`)
			So(html, ShouldContainSubstring, `<em>italicized</em>`)
		})

		Convey("should convert h1 and h2 to h3", func() {
			html := ConvertMarkdown(`# header1`)
			So(html, ShouldContainSubstring, `<h3>header1</h3>`)
			html = ConvertMarkdown(`## header2`)
			So(html, ShouldContainSubstring, `<h3>header2</h3>`)
		})

		Convey("should keep h4 to h6 as is", func() {
			html := ConvertMarkdown(`#### header1`)
			So(html, ShouldContainSubstring, `<h4>header1</h4>`)
			html = ConvertMarkdown(`##### header2`)
			So(html, ShouldContainSubstring, `<h5>header2</h5>`)
			html = ConvertMarkdown(`###### header3`)
			So(html, ShouldContainSubstring, `<h6>header3</h6>`)
		})
	})

}
