package models

import "time"

// Fields that can be updated by an UpdateAMR operation
var (
	AMRUpdateFields = []string{
		"title", "performer", "genre", "artist", "album_title",
		"song", "isrc", "is_match", "match_offset_seconds", "match_duration_seconds",
		"scan_offset_seconds", "scan_duration_seconds", "mute_offset_seconds",
		"mute_duration_seconds", "created_at", "updated_at", "song_length_seconds",
		"am_item_id",
	}
)

// AMR is the external facing representation of an audible magic response.
type AMR struct {
	AudibleMagicResponseID int64      `json:"id"`
	VodID                  int64      `json:"vod_id"`
	Title                  NullString `json:"title"`
	Performer              NullString `json:"performer"`
	Genre                  NullString `json:"genre"`
	Artist                 NullString `json:"artist"`
	AlbumTitle             NullString `json:"album_title"`
	Song                   NullString `json:"song"`
	Isrc                   NullString `json:"isrc"`
	IsMatch                bool       `json:"is_match"`
	MatchOffsetSeconds     NullInt64  `json:"match_offset_seconds"`
	MatchDurationSeconds   NullInt64  `json:"match_duration_seconds"`
	ScanOffsetSeconds      NullInt64  `json:"scan_offset_seconds"`
	ScanDurationSeconds    NullInt64  `json:"scan_duration_seconds"`
	MuteOffsetSeconds      NullInt64  `json:"mute_offset_seconds"`
	MuteDurationSeconds    NullInt64  `json:"mute_duration_seconds"`
	CreatedAt              time.Time  `json:"created_at"`
	UpdatedAt              time.Time  `json:"updated_at"`
	SongLength             NullInt64  `json:"song_length_seconds"`
	UnmutedAt              NullTime   `json:"unmuted_at"`
	AmItemID               NullString `json:"am_item_id"`
}

// AMRList represents a list of AMRs.
type AMRList []*AMR

// SpadeAttributes are the attributes in an AMR to send to spade
func (a *AMR) SpadeAttributes() map[string]interface{} {
	return map[string]interface{}{
		"amr_id":         a.AudibleMagicResponseID,
		"vod_id":         a.VodID,
		"muted_from":     a.MuteOffsetSeconds.Int64,
		"muted_duration": a.MuteDurationSeconds.Int64,
		"match_from":     a.MatchOffsetSeconds.Int64,
		"match_duration": a.MatchDurationSeconds.Int64,
		"am_item_id":     a.AmItemID.String,
		"isrc":           a.Isrc.String,
		"title":          a.Title.String,
		"performer":      a.Performer.String,
		"genre":          a.Genre.String,
		"artist":         a.Artist.String,
		"album_title":    a.AlbumTitle.String,
	}
}
