package models

import (
	"database/sql"
	"encoding/json"
	"fmt"
	"time"

	"github.com/lib/pq"
)

// NullInt64 shadows the sql.NullInt64 type.
type NullInt64 sql.NullInt64

// UnmarshalJSON checks for a null int and unmarshals it as necessary.
func (T *NullInt64) UnmarshalJSON(b []byte) error {
	var in interface{}

	if err := json.Unmarshal(b, &in); err != nil {
		return err
	}

	switch v := in.(type) {
	case int64:
		T.Valid, T.Int64 = true, v
	case int:
		T.Valid, T.Int64 = true, int64(v)
	case float64:
		T.Valid, T.Int64 = true, int64(v)
	case nil:
		T.Valid, T.Int64 = false, 0
	default:
		return fmt.Errorf("Unrecognized type for integer field: %T", v)
	}
	return nil
}

// MarshalJSON writes the string `null` if the NullInt64 is not valid,
// otherwise marshaling the int.
func (T NullInt64) MarshalJSON() (b []byte, err error) {
	b = []byte(`null`)
	if T.Valid {
		b, err = json.Marshal(T.Int64)
	}
	return
}

// NullTime a type to represent the states a nullable
// time field can be in. Can be converted to/from pq.NullTime
type NullTime struct {
	Present bool
	Time    *time.Time
}

// UnmarshalJSON checks for a null time and unmarshals it as necessary.
func (T *NullTime) UnmarshalJSON(b []byte) error {
	var in interface{}
	if err := json.Unmarshal(b, &in); err != nil {
		return err
	}
	T.Present = true
	switch v := in.(type) {
	case time.Time:
		T.Time = &v
	case string:
		if v == "" {
			T.Time = nil
			return nil
		}
		var t time.Time
		if err := json.Unmarshal(b, &t); err != nil {
			return fmt.Errorf("Error while parsing string as time: %v", err)
		}
		T.Time = &t
	case nil:
		T.Time = nil
	default:
		return fmt.Errorf("Unrecognized type for time field: %T", v)
	}
	return nil
}

// MarshalJSON writes the string `null` if the NullTime is not valid,
// otherwise marshaling the int.
func (T NullTime) MarshalJSON() (b []byte, err error) {
	b = []byte(`null`)
	if T.Present && T.Time != nil {
		b, err = json.Marshal(T.Time)
	}
	return
}

// Equals checks if 2 Nulltime objects are equivalent taking into account nulls.
func (T NullTime) Equals(other NullTime) bool {
	if T.Present != other.Present {
		return false
	}
	if T.Time == nil && other.Time != nil {
		return false
	}
	if T.Time != nil && other.Time == nil {
		return false
	}
	if !(*T.Time).Equal(*other.Time) {
		return false
	}
	return true
}

// Invalid tells if the NullTime is invalid according to pq.NullTime.Valid
func (T NullTime) Invalid() bool {
	return !T.Present || T.Time == nil
}

// AsPQ converts NullTime to pq.NullTime
func (T NullTime) AsPQ() pq.NullTime {
	if T.Time == nil {
		return pq.NullTime{Valid: false}
	}
	return pq.NullTime{Valid: true, Time: *T.Time}
}

// PQAsNullTime converts pq.NullTime to NullTime
func PQAsNullTime(t pq.NullTime) NullTime {
	if t.Valid {
		return NullTime{Present: true, Time: &t.Time}
	}
	return NullTime{Present: false, Time: nil}
}

// NullString shadows the sql.NullString type.
type NullString sql.NullString

// UnmarshalJSON checks for a null string and unmarshals it as necessary.
func (T *NullString) UnmarshalJSON(b []byte) error {
	var in interface{}
	if err := json.Unmarshal(b, &in); err != nil {
		return err
	}
	switch v := in.(type) {
	case string:
		T.Valid, T.String = true, v
	case nil:
		T.Valid, T.String = false, ""
	default:
		return fmt.Errorf("Unrecognized type for string field: %T", v)
	}
	return nil
}

// MarshalJSON writes the string `null` if the NullString is not valid,
// otherwise marshaling the int.
func (T NullString) MarshalJSON() (b []byte, err error) {
	b = []byte(`null`)
	if T.Valid {
		b, err = json.Marshal(&T.String)
	}
	return
}

// NullBool shadows the sql.NullBool type.
type NullBool sql.NullBool

// UnmarshalJSON checks for a null boolean and unmarshals it as necessary.
func (T *NullBool) UnmarshalJSON(b []byte) error {
	var in interface{}
	if err := json.Unmarshal(b, &in); err != nil {
		return err
	}
	switch v := in.(type) {
	case bool:
		T.Valid, T.Bool = true, v
	case nil:
		T.Valid, T.Bool = false, false
	default:
		return fmt.Errorf("Unrecognized type for boolean field: %T", v)
	}
	return nil
}

// MarshalJSON writes the string `null` if the NullBool is not valid,
// otherwise marshaling the int.
func (T NullBool) MarshalJSON() (b []byte, err error) {
	b = []byte(`null`)
	if T.Valid {
		b, err = json.Marshal(&T.Bool)
	}
	return
}
