package models

import (
	"encoding/json"
	"strconv"

	"code.justin.tv/vod/vinyl/errors"
)

// Thumbnail represents a vod's thumbnail
type Thumbnail struct {
	Path   string          `json:"path"`
	Offset *DynamicFloat32 `json:"offset"`
}

// Type returns the thumbnail's type
func (t Thumbnail) Type() string {
	if t.Offset == nil {
		return "custom"
	}
	return "generated"
}

// Validate makes sure the thumbnail is valid
func (t Thumbnail) Validate() error {
	if t.Path == "" {
		return errors.MissingParamError{ParamName: "path"}
	}
	return nil
}

// DynamicFloat32 a float32 type with more lenient JSON marshaling
type DynamicFloat32 float32

// UnmarshalJSON parse json input into a float32
func (T *DynamicFloat32) UnmarshalJSON(b []byte) error {
	var in interface{}

	if err := json.Unmarshal(b, &in); err != nil {
		return err
	}

	switch in.(type) {
	case float32:
		*T = DynamicFloat32(in.(float32))
	case float64:
		*T = DynamicFloat32(float32(in.(float64)))
	case string:
		inFloat, err := strconv.ParseFloat(in.(string), 32)
		if err != nil {
			return err
		}
		*T = DynamicFloat32(float32(inFloat))
	}

	return nil
}

// MarshalJSON marshal float32 into json
func (T DynamicFloat32) MarshalJSON() (b []byte, err error) {
	return json.Marshal(float32(T))
}

// Thumbnails a collection of thumbnails
type Thumbnails []*Thumbnail

// Dedupe removes duplicate thumbnails from the collection based on path
func (thumbnails Thumbnails) Dedupe() (deduped Thumbnails) {
	registry := map[string]bool{}
	for _, thumbnail := range thumbnails {
		if _, ok := registry[thumbnail.Path]; ok {
			continue
		}
		deduped = append(deduped, thumbnail)
		registry[thumbnail.Path] = true
	}
	return deduped
}

// DedupeFromExisting removes duplicates from the collection including existing
// thumbnails in the comparison.
func (thumbnails Thumbnails) DedupeFromExisting(existing Thumbnails) (deduped Thumbnails) {
	registry := map[string]bool{}
	for _, thumbnail := range existing {
		registry[thumbnail.Path] = true
	}
	for _, thumbnail := range thumbnails {
		if _, ok := registry[thumbnail.Path]; ok {
			continue
		}
		deduped = append(deduped, thumbnail)
		registry[thumbnail.Path] = true
	}
	return deduped
}
