package models

import (
	"fmt"
	"net/http"
	"net/url"
	"time"

	"code.justin.tv/vod/vinyl/errors"
	"code.justin.tv/vod/vinyl/utils"
)

// Fields that can be updated by an UpdateVod operation
var (
	VodUpdateFields = []string{
		"broadcast_id", "broadcast_type", "broadcaster_software", "created_by_id",
		"delete_at", "deleted", "description", "duration", "game", "language",
		"manifest", "notifications", "offset", "origin", "owner_id", "published_at", "show_formats", "source_archive_id",
		"started_on", "status", "tag_list", "thumbnails", "selected_thumbnail_path", "title", "updated_at", "uri", "views",
		"viewable", "viewable_at",
	}
)

const (
	// StatusRecorded is a terminal VOD status
	StatusRecorded = "recorded"

	// StatusRecording is a VOD status that is used during recording
	StatusRecording = "recording"

	// StatusUnprocessed defines the unprocessed VOD status
	StatusUnprocessed = "unprocessed"

	// StatusCreated is the first VOD status in the Upload workflow
	StatusCreated = "created"

	// StatusUnderReview is the status used when a VOD is being reviewed
	StatusUnderReview = "under_review"

	// StatusUploading is the status used when a VOD is busy uploading
	StatusUploading = "uploading"

	// StatusPendingTranscode is the status used when a VOD is waiting to be transcoded
	StatusPendingTranscode = "pending_transcode"

	// StatusFailed is the status used when VOD uploading has failed
	StatusFailed = "failed"

	// StatusTranscoding is the status used when a VOD is transcoding
	StatusTranscoding = "transcoding"
)

// Vod is the external user facing representation of a VOD.
type Vod struct {
	BroadcastID         int                `json:"broadcast_id"`
	BroadcastType       string             `json:"broadcast_type"`
	BroadcasterSoftware NullString         `json:"broadcaster_software"`
	CreatedAt           time.Time          `json:"created_at"`
	CreatedBy           NullInt64          `json:"created_by_id"`
	DeleteAt            NullTime           `json:"delete_at"`
	Deleted             NullBool           `json:"deleted"`
	Description         NullString         `json:"description"`
	DescriptionHTML     NullString         `json:"description_html"`
	Duration            int                `json:"duration"`
	Fps                 map[string]float64 `json:"fps"`
	Game                NullString         `json:"game"`
	ID                  int64              `json:"id"`
	Language            NullString         `json:"language"`
	Manifest            NullString         `json:"manifest"`
	Offset              int                `json:"offset"`
	Origin              NullString         `json:"origin"`
	OwnerID             int64              `json:"owner_id"`
	Resolutions         map[string]string  `json:"resolutions"`
	SourceArchiveID     NullInt64          `json:"source_archive_id"`
	StartedOn           time.Time          `json:"recorded_on"`
	Status              string             `json:"status"`
	TagList             string             `json:"tag_list"`
	Title               NullString         `json:"title"`
	TotalLength         int                `json:"total_length"`
	UpdatedAt           time.Time          `json:"updated_at"`
	URI                 string             `json:"uri"`
	Viewable            NullString         `json:"viewable"`
	ViewableAt          NullTime           `json:"viewable_at"`
	Views               int64              `json:"views"`
	PublishedAt         NullTime           `json:"published_at"`

	IncrementViewURL   string                            `json:"increment_view_count_url"`
	Qualities          []string                          `json:"qualities"`
	PreviewTemplate    string                            `json:"preview_template"`
	ShowFormats        map[string]map[string]interface{} `json:"show_formats"`
	ThumbnailTemplates interface{}                       `json:"thumbnail_templates"`
	Thumbnails         Thumbnails                        `json:"thumbnails"`
	SeekPreviewsURL    string                            `json:"seek_previews_url"`
	AnimatedPreviewURL string                            `json:"animated_preview_url"`
	Path               string                            `json:"path"`
	URL                string                            `json:"url"`

	// Fields associated with AMR & VOD Appeal
	AudibleMagicResponses AMRList         `json:"audible_magic_responses,omitempty"`
	VodAppeal             *VodAppeal      `json:"vod_appeal"`
	TrackAppeals          TrackAppealList `json:"track_appeals"`
	IsMuted               NullBool        `json:"is_muted"`
	CanCreateAppeal       NullBool        `json:"can_create_appeal"`
	ViewOnlyAppeal        NullBool        `json:"view_only_appeal"`
	HasPendingAppeal      NullBool        `json:"has_pending_appeal"`

	// VOD notifications
	NotificationSettings VodNotificationSettingsMap `json:"notifications,omitempty"`

	// TODO: Deprecate this
	APIID string `json:"api_id"`
}

// VodList is a list of Vods.
type VodList []*Vod

// HighlightInput contains the possible input parameters that go into a CreateHighlight operation.
type HighlightInput struct {
	VodID        int64      `json:"id"`
	Description  NullString `json:"description"`
	Game         NullString `json:"game"`
	Title        NullString `json:"title"`
	Language     NullString `json:"language"`
	StartSeconds int64      `json:"start_time"`
	EndSeconds   int64      `json:"end_time"`
	TagList      string     `json:"tag_list"`
	CreatedBy    NullInt64  `json:"created_by_id"`

	// TODO: remove this when migration is done. this is temporary so that we can
	// partition on the create highlight's id and check when the highlight is created
	HighlightID int64     `json:"highlight_id"`
	CreatedAt   time.Time `json:"created_at"`
}

// YoutubeExportInput is the input parameters for a Youtube Export.
type YoutubeExportInput struct {
	VodID       string `json:"vod_id"`
	Description string `json:"description"`
	Title       string `json:"title"`
	TagList     string `json:"tag_list"`
	Private     bool   `json:"private"`
	DoSplit     bool   `json:"do_split"`
}

// VodNotificationSettingsInput contains the possible input parameters for notification
// settings that are part of `VodUpdateInput`.
type VodNotificationSettingsInput struct {
	CustomText NullString `json:"custom_text"`
	Enabled    bool       `json:"enabled"`
	SentAt     NullTime   `json:"sent_at"`
}

// VodNotificationSettingsMapInput contains the map of possible VodNotificationSettingsInput
type VodNotificationSettingsMapInput struct {
	Email       *VodNotificationSettingsInput `json:"email,omitempty"`
	ChannelFeed *VodNotificationSettingsInput `json:"channel_feed,omitempty"`
}

// VodUpdateInput contains the possible input parameters that go into a UpdateVod operation.
type VodUpdateInput struct {
	BroadcastID           NullInt64  `json:"broadcast_id"`
	BroadcastType         NullString `json:"broadcast_type"`
	BroadcasterSoftware   NullString `json:"broadcaster_software"`
	CreatedBy             NullInt64  `json:"created_by_id"`
	DeleteAt              NullInt64  `json:"delete_at"`
	Deleted               NullBool   `json:"deleted"`
	Description           NullString `json:"description"`
	Duration              NullInt64  `json:"duration"`
	Game                  NullString `json:"game"`
	Language              NullString `json:"language"`
	Manifest              NullString `json:"manifest"`
	Offset                NullInt64  `json:"offset"`
	Origin                NullString `json:"origin"`
	OwnerID               NullInt64  `json:"owner_id"`
	SelectedThumbnailPath NullString `json:"selected_thumbnail_path"`
	SourceArchiveID       NullInt64  `json:"source_archive_id"`
	StartedOn             NullInt64  `json:"started_on"`
	Status                NullString `json:"status"`
	Thumbnails            NullString `json:"thumbnails"`
	Title                 NullString `json:"title"`
	UpdatedAt             time.Time  `json:"updated_at"`
	URI                   NullString `json:"uri"`
	Views                 NullInt64  `json:"views"`

	TagList     NullString                        `json:"tag_list"`
	ShowFormats map[string]map[string]interface{} `json:"show_formats"`

	ViewableAt  NullTime   `json:"viewable_at"`
	Viewable    NullString `json:"viewable"`
	PublishedAt NullTime   `json:"published_at"`

	Notifications VodNotificationSettingsMapInput `json:"notifications"`
}

// VodFilters is a struct to hold vod query filters
type VodFilters struct {
	IncludeDeleted    bool
	IncludePrivate    bool
	IncludeProcessing bool
}

// VodOwner is a struct to hold vod owner information and vod information
type VodOwner struct {
	OwnerID    string    `json:"owner_id"`
	OwnerLogin *string   `json:"owner_login"`
	Status     string    `json:"status"`
	Duration   int       `json:"duration"`
	StartedOn  time.Time `json:"recorded_on"`
}

// VodFilterNone returns a set of all disabled filters
func VodFilterNone() *VodFilters {
	return &VodFilters{
		IncludeDeleted:    true,
		IncludePrivate:    true,
		IncludeProcessing: true,
	}
}

// VodFilterDeleted will filter out deleted vods
func VodFilterDeleted() *VodFilters {
	return &VodFilters{
		IncludeDeleted:    false,
		IncludePrivate:    true,
		IncludeProcessing: true,
	}
}

// ParseFilters parses filter options out of URL query values
func ParseFilters(values url.Values) (*VodFilters, error) {
	includeDeleted, err := utils.ParseBool(values.Get("include_deleted"))
	if err != nil {
		return nil, errors.InvalidValueError{ParamName: "include_deleted", Value: values.Get("include_deleted")}
	}

	includePrivate, err := utils.ParseBool(values.Get("include_private"))
	if err != nil {
		return nil, errors.InvalidValueError{ParamName: "include_private", Value: values.Get("include_private")}
	}

	includeProcessing, err := utils.ParseBool(values.Get("include_processing"))
	if err != nil {
		return nil, errors.InvalidValueError{ParamName: "include_processing", Value: values.Get("include_processing")}
	}

	return &VodFilters{
		IncludeDeleted:    includeDeleted,
		IncludePrivate:    includePrivate,
		IncludeProcessing: includeProcessing,
	}, nil
}

// Watchable returns true if the VOD is active and published.
func (v *Vod) Watchable() bool {
	return (!v.Deleted.Valid || !v.Deleted.Bool) &&
		(!v.Viewable.Valid || v.Viewable.String != "private") &&
		(v.Status == StatusRecording || v.Status == StatusRecorded)
}

// SpadeAttributes are the attributes in a vod to send to spade
func (v *Vod) SpadeAttributes(r *http.Request) map[string]interface{} {
	var viewableAt float64
	if v.ViewableAt.Present {
		viewableAt = float64(v.ViewableAt.Time.Unix())
	}
	var clientRowID string
	if r != nil {
		clientRowID = r.Header.Get("Twitch-Client-Row-Id")
	}
	return map[string]interface{}{
		"vod_id":                 fmt.Sprintf("%d", v.ID),
		"channel_id":             v.OwnerID,
		"created_by_id":          v.CreatedBy,
		"duration":               v.Duration / 60,
		"duration_seconds":       v.Duration,
		"vod_type":               v.BroadcastType,
		"viewable_at":            viewableAt,
		"viewable":               v.Viewable.String,
		"status":                 v.Status,
		"title":                  v.Title.String,
		"deleted":                v.Deleted.Bool,
		"game":                   v.Game.String,
		"description":            v.Description.String,
		"language":               v.Language.String,
		"using_custom_thumbnail": false, // todo: fix this
		"tags":          v.TagList,
		"published_at":  v.PublishedAt,
		"created_at":    v.CreatedAt.Unix(),
		"updated_at":    v.UpdatedAt.Unix(),
		"client_row_id": clientRowID,
	}
}
