# Primary application ASG and ELB

resource "template_file" "app_user_data" {
  template = "${file("${path.module}/asg_puppetizer.template")}"
  vars {
    hostname_prefix = "${var.hostname}-aws"
    domain = "${lookup(var.domain, var.environment)}"
    cluster = "${var.name}-${var.environment}"
    environment = "${var.environment}"
    canary = "false"
  }
}

resource "aws_launch_configuration" "app_conf" {
  name_prefix = "${var.name}-${var.environment}-app-"
  image_id = "${var.ami}"
  instance_type = "${var.instance_type}"
  security_groups = ["${terraform_remote_state.account.output.twitch_subnets_sg}"]
  user_data = "${template_file.app_user_data.rendered}"
  iam_instance_profile = "${aws_iam_instance_profile.app.id}"

  root_block_device {
    volume_size = "${var.root_volume_size}"
  }

  lifecycle {
    create_before_destroy = true
  }
}

resource "aws_autoscaling_group" "app_asg" {
  name = "${var.name}-${var.environment}-app"
  vpc_zone_identifier = ["${terraform_remote_state.account.output.service_a}",
                          "${terraform_remote_state.account.output.service_b}",
                          "${terraform_remote_state.account.output.service_c}"]
  max_size = "${var.max_size}"
  min_size = "${var.min_size}"
  health_check_grace_period = 600
  health_check_type = "EC2"
  load_balancers = ["${aws_elb.internal_app_elb.name}"]
  launch_configuration = "${aws_launch_configuration.app_conf.name}"

  tag {
    key = "Name"
    value = "${var.name}"
    propagate_at_launch = true
  }
  tag {
    key = "Owner"
    value = "${var.owner}"
    propagate_at_launch = true
  }
  tag {
    key = "Project"
    value = "${var.project}"
    propagate_at_launch = true
  }
}

resource "aws_autoscaling_policy" "app_asg_scaleout_policy" {
  name = "${var.name}-${var.environment}-app-scaleout-policy"
  scaling_adjustment = 1
  adjustment_type = "ChangeInCapacity"
  cooldown = 1200
  autoscaling_group_name = "${aws_autoscaling_group.app_asg.name}"
}

resource "aws_cloudwatch_metric_alarm" "asg_high_cpu" {
  alarm_name = "${var.name}-${var.environment}-app-high-cpu"
  comparison_operator = "GreaterThanOrEqualToThreshold"
  evaluation_periods = "5"
  metric_name = "CPUUtilization"
  namespace = "AWS/EC2"
  period = "60"
  statistic = "Average"
  threshold = "50"
  dimensions = {AutoScalingGroupName = "${aws_autoscaling_group.app_asg.id}"}
  alarm_actions = ["${aws_autoscaling_policy.app_asg_scaleout_policy.arn}"]
}

resource "aws_autoscaling_policy" "app_asg_scalein_policy" {
  name = "${var.name}-${var.environment}-asg-scalein-policy"
  scaling_adjustment = -1
  adjustment_type = "ChangeInCapacity"
  cooldown = 900
  autoscaling_group_name = "${aws_autoscaling_group.app_asg.name}"
}

resource "aws_cloudwatch_metric_alarm" "asg_low_cpu" {
  alarm_name = "${var.name}-${var.environment}-asg-low-cpu"
  comparison_operator = "LessThanOrEqualToThreshold"
  evaluation_periods = "15"
  metric_name = "CPUUtilization"
  namespace = "AWS/EC2"
  period = "60"
  statistic = "Average"
  threshold = "10"
  dimensions = {AutoScalingGroupName = "${aws_autoscaling_group.app_asg.id}"}
  alarm_actions = ["${aws_autoscaling_policy.app_asg_scalein_policy.arn}"]
}

module "asg_dns" {
  source = "git::git+ssh://git@git-aws.internal.justin.tv/web/asg-r53-lambda.git//modules/notifications"
  topic_name = "${var.name}_${var.environment}_topic"
  asg_name = "${aws_autoscaling_group.app_asg.name}"
  lambda_arn = "${terraform_remote_state.asg_lambda.output.asg_lambda_arn}"
}

resource "aws_elb" "internal_app_elb" {
  name = "${var.name}-${var.environment}-app"
  security_groups = ["${terraform_remote_state.account.output.twitch_subnets_sg}"]
  subnets = ["${split(",", "${terraform_remote_state.account.output.service_subnets}")}"]
  internal = true
  cross_zone_load_balancing = true
  connection_draining = true

  listener {
    instance_port = 8000
    instance_protocol = "http"
    lb_port = 80
    lb_protocol = "http"
  }

  health_check {
    healthy_threshold = 3
    unhealthy_threshold = 5
    timeout = 10
    target = "HTTP:8000/debug/running"
    interval = 30
  }

  tags {
    Name = "${var.name}-${var.environment}-app"
    Owner = "${var.owner}"
    Project = "${var.project}"
  }
}

module "base" {
    source = "git::git+ssh://git@git-aws.internal.justin.tv/release/terraform.git//base"
}

provider "aws" {
  alias = "dns"
  access_key = "${var.dns_access_key}"
  secret_key = "${var.dns_secret_key}"
  region = "us-west-2"
}

resource "aws_route53_record" "elb_dns" {
  provider = "aws.dns"
  zone_id = "${module.base.twitch_r53}"
  name = "${var.name}-internal.${var.environment}.us-west2.twitch.tv"
  type = "CNAME"
  ttl = "300"
  records = ["${aws_elb.internal_app_elb.dns_name}"]
}
