package main

import (
	"fmt"
	"math/rand"
	"net/http"
	"os"
	"time"

	apertureredis "code.justin.tv/businessviewcount/aperture/internal/clients/redis"

	"code.justin.tv/businessviewcount/aperture/config"
	"code.justin.tv/businessviewcount/aperture/internal/apertureserver"
	"code.justin.tv/businessviewcount/aperture/internal/clients/blender"
	"code.justin.tv/businessviewcount/aperture/internal/clients/memcached"
	"code.justin.tv/businessviewcount/aperture/internal/clients/multiplex"
	"code.justin.tv/businessviewcount/aperture/internal/clients/pubsub"
	"code.justin.tv/businessviewcount/aperture/internal/clients/secrets"
	"code.justin.tv/businessviewcount/aperture/internal/clients/spade"
	"code.justin.tv/businessviewcount/aperture/internal/clients/stats"
	"code.justin.tv/businessviewcount/aperture/internal/clients/viewcount"
	"code.justin.tv/businessviewcount/aperture/internal/fetcher"
	"code.justin.tv/businessviewcount/aperture/internal/util"
	"code.justin.tv/businessviewcount/aperture/rpc/aperture"

	"github.com/twitchtv/twirp"
	"goji.io/pat"

	log "github.com/sirupsen/logrus"
	statsdhook "github.com/twitchtv/twirp/hooks/statsd"
	goji "goji.io"
)

func main() {
	port := os.Getenv("PORT")
	if port == "" {
		log.Fatal("aperture: no port found in env")
		return
	}

	env := os.Getenv("ENVIRONMENT")
	if env == "" {
		log.Fatal("aperture: no environment found in env")
		return
	}

	// ADVAX-1061 - log freezes
	envUUID := createUUID()
	log.Infof("Setting Host ID to %s", envUUID)
	os.Setenv("HOST-ID", envUUID)

	conf := &config.Config{
		Environment: env,
	}

	err := conf.Load()
	if err != nil {
		log.Fatal("aperture: could not load config: ", err)
		return
	}

	secretManager, err := secrets.NewManager()
	if err != nil {
		log.Fatal("aperture: could not create secrets manager: ", err)
		return
	}

	config.SetupRollbarLogging(secretManager,
		conf.RollbarTokenSecretName.Get(),
		conf.RollbarTokenSecretKey.Get(),
		env)

	statsdClient, err := stats.NewClient(conf.StatsdHost.Get(), env)
	if err != nil {
		log.Fatal("aperture: could not create statsd client: ", err)
		return
	}

	spadeClient, err := spade.NewClient(conf.SpadeHost.Get(), statsdClient)
	if err != nil {
		log.Fatal("aperture: could not create spade client: ", err)
		return
	}

	pubsubClient, err := pubsub.NewClient(conf.PubsubHost.Get(), conf.SpadeHost.Get())
	if err != nil {
		log.Fatal("aperture: could not create pubsub client: ", err)
		return
	}

	cache := memcached.NewCache(
		conf.CacheAddress.Get(),
		int(conf.CacheMaxIdleConns.Get()),
		conf.CachePollInterval.Get(),
		conf.CacheTimeout.Get(),
		conf.CacheExpirationDuration,
	)

	viewcountClient, err := viewcount.NewClient(conf.ViewcountHost.Get(), cache, statsdClient)
	if err != nil {
		log.Fatal("aperture: could not create viewcount client: ", err)
		return
	}

	redisCli, err := apertureredis.NewClient(conf.RedisCacheAddress.Get(), statsdClient, "service")
	if err != nil {
		log.Fatal("aperture: could not connect to redis: ", err)
		return
	}

	multiplexClient := multiplex.NewClient(conf.MultiplexHost.Get(), statsdClient)
	blenderClient := blender.NewClient(conf.BlenderHost.Get(), statsdClient)
	fetcherClient := fetcher.NewClient(&fetcher.Params{
		Blender:     blenderClient,
		Viewcount:   viewcountClient,
		Cache:       cache,
		RatioBuffer: util.NewRatioBuffer(),
		Stats:       statsdClient,
		RedisCli:    redisCli,
	})

	server := &apertureserver.Server{
		Config:    conf,
		Statsd:    statsdClient,
		Spade:     spadeClient,
		Multiplex: multiplexClient,
		Pubsub:    pubsubClient,
		Fetcher:   fetcherClient,
		Blender:   blenderClient,
		Redis:     redisCli,
	}

	statsdHook := statsdhook.NewStatsdServerHooks(statsdClient.Statter)
	hooks := twirp.ChainHooks(statsdHook)
	twirpHandler := aperture.NewApertureServer(server, hooks)

	addr := fmt.Sprintf(":%s", port)

	mux := goji.NewMux()
	mux.Use(util.PanicRecovery)
	mux.Handle(pat.Post(aperture.AperturePathPrefix+"*"), twirpHandler)
	mux.HandleFunc(pat.Get("/health"), util.HealthCheck)

	log.Infof("aperture: server listening on port %s", addr)

	if err := http.ListenAndServe(addr, mux); err != nil {
		log.Error("server error: ", err)
	}
}

func createUUID() string {
	charset := "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789"

	seededRand := rand.New(rand.NewSource(time.Now().UnixNano()))

	b := make([]byte, 10)
	for i := range b {
		b[i] = charset[seededRand.Intn(len(charset))]
	}

	return string(b)
}
