package config

import (
	"fmt"
	"time"

	"code.justin.tv/hygienic/distconf"
	"code.justin.tv/hygienic/paramstoreconf"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/ssm"
	"github.com/pkg/errors"
	log "github.com/sirupsen/logrus"
)

const (
	// DevelopmentEnv is the development environment name
	DevelopmentEnv = "development"

	// StagingEnv is the staging environment name
	StagingEnv = "staging"

	// ProductionEnv is the production environment name
	ProductionEnv = "production"

	defaultCacheExpirationDuration = 10 * time.Minute
	defaultCacheMaxIdleConns       = 1000
	defaultCachePollInterval       = time.Minute
	defaultCacheTimeout            = 1000 * time.Millisecond
	defaultDistconfRefreshInterval = 30 * time.Second
)

// Config holds values for aperture server configuration
type Config struct {
	Environment            string
	RollbarTokenSecretName *distconf.Str
	RollbarTokenSecretKey  *distconf.Str
	StatsdHost             *distconf.Str
	BlenderHost            *distconf.Str
	MultiplexHost          *distconf.Str
	PubsubHost             *distconf.Str
	ViewcountHost          *distconf.Str
	ApertureHost           *distconf.Str
	SpadeHost              *distconf.Str

	CacheAddress            *distconf.Str
	CacheExpirationDuration *distconf.Duration
	CacheMaxIdleConns       *distconf.Int
	CachePollInterval       *distconf.Duration
	CacheTimeout            *distconf.Duration

	LoggingChannelsWhitelist *distconf.Str

	RedisCacheAddress *distconf.Str
}

// Load loads the appropriate config values into the config struct. This should be
// done when config is required before creating an aperture server
func (c *Config) Load() error {
	err := validateEnvironment(c.Environment)
	if err != nil {
		return err
	}

	dconf, err := createDistconf(c.Environment)
	if err != nil {
		return err
	}

	c.RollbarTokenSecretName = dconf.Str("rollbar.secret.name", "")
	if c.RollbarTokenSecretName.Get() == "" {
		return errors.New("aperture: could not find rollbar secret name in config")
	}

	c.RollbarTokenSecretKey = dconf.Str("rollbar.secret.key", "")
	if c.RollbarTokenSecretKey.Get() == "" {
		return errors.New("aperture: could not find rollbar secret key in config")
	}

	c.StatsdHost = dconf.Str("statsd.host", "")
	if c.StatsdHost.Get() == "" {
		return errors.New("aperture: could not find statsd host in config")
	}

	c.PubsubHost = dconf.Str("pubsub.host", "")
	if c.PubsubHost.Get() == "" {
		return errors.New("aperture: could not find pubsub host in config")
	}

	c.ViewcountHost = dconf.Str("viewcount.host", "")
	if c.ViewcountHost.Get() == "" {
		return errors.New("aperture: could not find viewcount host in config")
	}

	c.BlenderHost = dconf.Str("blender.host", "")
	if c.BlenderHost.Get() == "" {
		return errors.New("aperture: could not find blender host in config")
	}

	c.MultiplexHost = dconf.Str("multiplex.host", "")
	if c.MultiplexHost.Get() == "" {
		return errors.New("aperture: could not find multiplex host in config")
	}

	c.ApertureHost = dconf.Str("aperture.host", "")
	if c.ApertureHost.Get() == "" {
		return errors.New("aperture: could not find aperture host in config")
	}

	c.CacheAddress = dconf.Str("cache.address", "")
	if c.CacheAddress.Get() == "" {
		return errors.New("aperture: could not find ElastiCache address in config")
	}

	c.SpadeHost = dconf.Str("spade.host", "")
	if c.SpadeHost.Get() == "" {
		return errors.New("aperture: could not find spade host in config")
	}

	c.RedisCacheAddress = dconf.Str("redis.host", "")
	if c.RedisCacheAddress.Get() == "" {
		return errors.New("aperture: could not find redis host in config")
	}

	c.CacheExpirationDuration = dconf.Duration("cache.expiration_duration", defaultCacheExpirationDuration)
	c.CacheMaxIdleConns = dconf.Int("cache.max_idle_conns", defaultCacheMaxIdleConns)
	c.CachePollInterval = dconf.Duration("cache.poll_interval", defaultCachePollInterval)
	c.CacheTimeout = dconf.Duration("cache.timeout", defaultCacheTimeout)

	c.LoggingChannelsWhitelist = dconf.Str("logging.whitelist", "")

	return nil
}

// createDistconf creates a distConf struct with a json reader and parameter store reader, and refreshes the config periodically
func createDistconf(env string) (*distconf.Distconf, error) {
	logger := &configLogger{}

	configFilename := fmt.Sprintf("config/%s.json", env)
	var baseConfig distconf.JSONConfig
	if err := baseConfig.RefreshFile(configFilename); err != nil {
		return nil, err
	}

	awsConfig := aws.NewConfig().WithRegion("us-west-2")
	sess, err := session.NewSession(awsConfig)
	if err != nil {
		return nil, errors.Wrap(err, "config: failed to initialize new aws session")
	}

	paramStoreKeys := &paramstoreconf.ParameterStoreConfiguration{
		SSM:         ssm.New(sess),
		Prefix:      "configurations",
		Team:        "cb",
		Environment: env,
		Service:     "aperture",
		Logger:      logger,
	}

	config := &distconf.Distconf{
		Readers: []distconf.Reader{
			&baseConfig,
			paramStoreKeys,
		},
	}

	configRefresher := distconf.Refresher{
		ToRefresh: paramStoreKeys,
		WaitTime:  config.Duration("distconf.refresh_interval", defaultDistconfRefreshInterval),
	}

	go func() {
		err := configRefresher.Start()
		if err != nil {
			log.WithError(err).Error("aperture: failed to start the config refresher")
		}
	}()
	return config, nil
}

func validateEnvironment(env string) error {
	if env == "" || (env != ProductionEnv && env != StagingEnv && env != DevelopmentEnv) {
		return fmt.Errorf("aperture: invalid environment '%s' in config loader", env)
	}

	return nil
}
