package apertureserver

import (
	"context"
	"fmt"
	"time"

	"code.justin.tv/businessviewcount/aperture/config"
	"code.justin.tv/businessviewcount/aperture/internal/clients/blender"
	"code.justin.tv/businessviewcount/aperture/internal/clients/multiplex"
	"code.justin.tv/businessviewcount/aperture/internal/clients/pubsub"
	apertureredis "code.justin.tv/businessviewcount/aperture/internal/clients/redis"
	"code.justin.tv/businessviewcount/aperture/internal/clients/spade"
	"code.justin.tv/businessviewcount/aperture/internal/clients/stats"
	"code.justin.tv/businessviewcount/aperture/internal/fetcher"
	"code.justin.tv/businessviewcount/aperture/internal/util"

	"github.com/pkg/errors"
	"github.com/twitchtv/twirp"

	pb "code.justin.tv/businessviewcount/aperture/rpc/aperture"
)

// Server implements the Aperture service
type Server struct {
	Config    *config.Config
	Statsd    stats.StatSender
	Spade     spade.Injector
	Pubsub    pubsub.Publisher
	Multiplex multiplex.Multiplex
	Fetcher   fetcher.Fetcher
	Blender   blender.Internal
	Redis     apertureredis.RedisClient
}

// GetViewcountForChannel returns view counts for a single channel
func (s *Server) GetViewcountForChannel(ctx context.Context, req *pb.GetViewcountForChannelReq) (*pb.GetViewcountForChannelResp, error) {
	ctx, cancel := context.WithTimeout(ctx, 2*time.Second)
	defer cancel()

	channelID := req.GetChannelID()
	views, err := s.Fetcher.FetchViewcountForChannel(ctx, channelID)
	if err != nil {
		if terr, ok := err.(twirp.Error); ok {
			return nil,
				twirp.NewError(terr.Code(),
					fmt.Sprintf("failed to get viewcount from viewcount-api: %v", terr.Msg())).WithMeta("channelID", channelID)
		}

		switch err {
		case fetcher.ErrInvalidChannelID:
			return nil, twirp.NewError(twirp.InvalidArgument, "invalid channel id provided").WithMeta("channelID", channelID)
		case fetcher.ErrWorkerPool:
			return nil, twirp.NewError(twirp.Internal, "failed to acquire worker").WithMeta("channelID", channelID)
		default:
			return nil, errors.Wrap(err, "failed to get viewcount from viewcount-api")
		}
	}

	return &pb.GetViewcountForChannelResp{
		Views: views,
	}, nil
}

// FreezeChannel freezes viewcounts for a single channel for a specified period of time
func (s *Server) FreezeChannel(ctx context.Context, channelProps *pb.FreezeChannelReq) (*pb.FreezeChannelResp, error) {
	ctx, cancel := context.WithTimeout(ctx, 1*time.Second)
	defer cancel()

	return s.Fetcher.FreezeChannel(ctx, channelProps)
}

// GetViewcountForAllChannels returns view counts for all live channels
func (s *Server) GetViewcountForAllChannels(ctx context.Context, req *pb.GetViewcountForAllChannelsReq) (*pb.GetViewcountForAllChannelsResp, error) {
	ctx, cancel := context.WithTimeout(ctx, 5*time.Second)
	defer cancel()

	filteredViewcounts, err := s.Fetcher.FetchViewcountsForAll(ctx)
	if err != nil {
		if terr, ok := err.(twirp.Error); ok {
			return nil,
				twirp.NewError(terr.Code(),
					fmt.Sprintf("failed to get viewcount from viewcount-api: %v", terr.Msg()))
		}

		switch err {
		case fetcher.ErrWorkerPool:
			return nil, twirp.NewError(twirp.Internal, "failed to acquire worker")
		default:
			return nil, errors.Wrap(err, "failed to get viewcount from viewcount-api")
		}
	}

	return &pb.GetViewcountForAllChannelsResp{
		Views: filteredViewcounts,
	}, nil
}

// UpdateSpadeViewcount gets view counts for all live channels, then writes the counts in batches to spade
func (s *Server) UpdateSpadeViewcount(ctx context.Context, req *pb.UpdateSpadeViewcountReq) (resp *pb.UpdateSpadeViewcountResp, err error) {
	// Get viewcounts for all channels from viewcount-api
	filteredViewcounts, err := s.Fetcher.FetchViewcountsForAll(ctx)
	s.Statsd.SendGauge("aperture.spade_filtered_views_size", int64(len(filteredViewcounts)))
	if err != nil {
		return nil, errors.Wrap(err, "failed to get viewcounts from viewcount-api when updating spade")
	}

	multiplexResp, multiplexErr := s.Multiplex.StreamMinuteBroadcasts(ctx)
	s.Statsd.SendGauge("aperture.spade_multiplex_mb_size", int64(len(multiplexResp)))

	if multiplexErr != nil {
		return nil, errors.Wrap(multiplexErr, "failed to get minute broadcast from multiplex when updating spade")
	}

	// Convert filtered view counts and minute broadcast data to spade properties model
	spadeEvents := util.ConvertViewcountsToSpade(filteredViewcounts, multiplexResp)
	s.Statsd.SendGauge("aperture.spade_events_size", int64(len(spadeEvents.ChannelConcurrents)))
	s.Statsd.SendGauge("aperture.global_concurrents", int64(spadeEvents.GlobalConcurrents["total"]))

	// Send all events to spade. The spade client will handle batching and concurrency
	s.Spade.SendChannelConcurrents(context.Background(), s.Config.Environment, spadeEvents)

	return &pb.UpdateSpadeViewcountResp{}, nil
}

// UpdatePubsubViewcount gets view counts for all live channels, then sends those counts to pubsub
func (s *Server) UpdatePubsubViewcount(ctx context.Context, req *pb.UpdatePubsubViewcountReq) (resp *pb.UpdatePubsubViewcountResp, err error) {
	filteredViewcounts, err := s.Fetcher.FetchViewcountsForAll(ctx)
	s.Statsd.SendGauge("aperture.pubsub_filtered_views_size", int64(len(filteredViewcounts)))
	if err != nil {
		return nil, errors.Wrap(err, "failed to get viewcounts from viewcount-api when updating pubsub")
	}

	// Get additional minute broadcast data from multiplex
	mbData, err := s.Multiplex.StreamMinuteBroadcasts(ctx)
	s.Statsd.SendGauge("aperture.pubsub_multiplex_mb_size", int64(len(mbData)))
	if err != nil {
		return nil, errors.Wrap(err, "failed to get minute broadcast from multiplex when updating pubsub")
	}

	pubsubEvents := util.ConvertViewcountsToPubsub(filteredViewcounts, mbData)
	s.Statsd.SendGauge("aperture.pubsub_events_size", int64(len(pubsubEvents)))

	s.Pubsub.PublishChannelConcurrents(ctx, s.Config.Environment, pubsubEvents)

	return &pb.UpdatePubsubViewcountResp{}, nil
}
