package apertureserver_test

import (
	"context"
	"errors"
	"testing"

	"code.justin.tv/businessviewcount/aperture/config"
	"code.justin.tv/businessviewcount/aperture/internal/apertureserver"
	"code.justin.tv/businessviewcount/aperture/internal/mocks"
	"github.com/stretchr/testify/mock"

	pb "code.justin.tv/businessviewcount/aperture/rpc/aperture"
	"code.justin.tv/video/usherapi/rpc/usher"
	log "github.com/sirupsen/logrus"
	. "github.com/smartystreets/goconvey/convey"
)

type mockServerParams struct {
	Config    *config.Config
	Statsd    *mocks.StatSender
	Spade     *mocks.Injector
	Pubsub    *mocks.Publisher
	Fetcher   *mocks.Fetcher
	Blender   *mocks.Internal
	Multiplex *mocks.Multiplex
}

func serverMocks() mockServerParams {
	config := &config.Config{}
	statsd := &mocks.StatSender{}
	spade := &mocks.Injector{}
	pubsub := &mocks.Publisher{}
	fetcher := &mocks.Fetcher{}
	multiplex := &mocks.Multiplex{}
	blender := &mocks.Internal{}

	return mockServerParams{
		Config:    config,
		Statsd:    statsd,
		Spade:     spade,
		Pubsub:    pubsub,
		Fetcher:   fetcher,
		Multiplex: multiplex,
		Blender:   blender,
	}
}

func mockServer(params mockServerParams) *apertureserver.Server {
	return &apertureserver.Server{
		Config:    params.Config,
		Statsd:    params.Statsd,
		Spade:     params.Spade,
		Pubsub:    params.Pubsub,
		Fetcher:   params.Fetcher,
		Multiplex: params.Multiplex,
		Blender:   params.Blender,
	}
}

func TestGetViewcountForChannel(t *testing.T) {
	log.SetLevel(log.PanicLevel)
	channelID := "123456"
	mocks := serverMocks()
	server := mockServer(mocks)
	ctx := context.Background()

	singleChannelResp := &pb.Viewcount{
		Count:           10,
		CountUnfiltered: 10,
	}

	Convey("Given a valid request to get view counts for a single channel", t, func() {

		Convey("When the fetcher fails", func() {
			mocks.Fetcher.On("FetchViewcountForChannel", mock.Anything, channelID).Once().Return(nil, errors.New("api error"))

			Convey("The request should return an error", func() {
				req := &pb.GetViewcountForChannelReq{
					ChannelID: channelID,
				}

				_, err := server.GetViewcountForChannel(ctx, req)
				So(err, ShouldNotBeNil)
			})
		})

		Convey("When the fetcher succeeds", func() {
			mocks.Fetcher.On("FetchViewcountForChannel", mock.Anything, channelID).Once().Return(singleChannelResp, nil)

			Convey("The request should return viewcount", func() {
				req := &pb.GetViewcountForChannelReq{
					ChannelID: channelID,
				}

				resp, err := server.GetViewcountForChannel(ctx, req)
				So(err, ShouldBeNil)
				So(resp.Views.Count, ShouldEqual, 10)
			})
		})
	})
}

func TestGetViewcountForAllChannels(t *testing.T) {
	log.SetLevel(log.PanicLevel)
	allChannelsResp := map[string]*pb.Viewcount{
		"123456": {Count: 10},
		"222222": {Count: 6},
	}
	mocks := serverMocks()
	server := mockServer(mocks)
	ctx := context.Background()

	Convey("Given a valid request to get view counts for a all channels", t, func() {

		Convey("When the fetcher fails", func() {
			mocks.Fetcher.On("FetchViewcountsForAll", mock.Anything).Once().Return(nil, errors.New("api error"))

			Convey("The request should return an error", func() {
				req := &pb.GetViewcountForAllChannelsReq{}

				_, err := server.GetViewcountForAllChannels(ctx, req)
				So(err, ShouldNotBeNil)
			})
		})

		Convey("When the fetcher succeeds", func() {
			mocks.Fetcher.On("FetchViewcountsForAll", mock.Anything).Once().Return(allChannelsResp, nil)

			Convey("The request should succeed with view counts", func() {
				req := &pb.GetViewcountForAllChannelsReq{}

				resp, err := server.GetViewcountForAllChannels(ctx, req)
				So(err, ShouldBeNil)
				So(resp.Views["123456"].Count, ShouldEqual, 10)
				So(resp.Views["222222"].Count, ShouldEqual, 6)
			})
		})
	})
}

func TestFreezeChannel(t *testing.T) {
	log.SetLevel(log.PanicLevel)
	mocks := serverMocks()
	server := mockServer(mocks)
	ctx := context.Background()

	Convey("Given a valid request to freeze channel", t, func() {
		mocks.Fetcher.On("FreezeChannel", mock.Anything, mock.Anything).Once().Return(&pb.FreezeChannelResp{}, nil)

		Convey("The request should succeed with no op", func() {
			req := &pb.FreezeChannelReq{}

			_, err := server.FreezeChannel(ctx, req)
			So(err, ShouldBeNil)
		})
	})
}

func TestUpdateSpadeViewcount(t *testing.T) {
	log.SetLevel(log.PanicLevel)
	allChannelsResp := map[string]*pb.Viewcount{
		"123456": {Count: 10},
		"222222": {Count: 6},
	}

	mbResp := []*usher.StreamMinuteBroadcast{
		{
			Channel:    "test_channel1",
			ChannelId:  int64(123456),
			ContentId:  "123456",
			CustomerId: "twitch",
			Platform:   "unknown_rtmp",
		},
		{
			Channel:    "test_channel2",
			ChannelId:  int64(222222),
			ContentId:  "222222",
			CustomerId: "twitch",
			Platform:   "unknown_rtmp",
		},
	}

	mocks := serverMocks()
	server := mockServer(mocks)
	ctx := context.Background()

	Convey("When a valid request is sent to update spade values", t, func() {

		Convey("When the fetcher fails", func() {
			mocks.Fetcher.On("FetchViewcountsForAll", ctx).Once().Return(nil, errors.New("api error"))
			mocks.Multiplex.AssertNotCalled(t, "StreamMinuteBroadcasts", ctx)
			mocks.Statsd.On("SendGauge", mock.Anything, mock.Anything).Once()

			Convey("The request should return an error", func() {
				req := &pb.UpdateSpadeViewcountReq{}

				_, err := server.UpdateSpadeViewcount(ctx, req)
				So(err, ShouldNotBeNil)
				mocks.Fetcher.AssertExpectations(t)
				mocks.Multiplex.AssertExpectations(t)
				mocks.Statsd.AssertExpectations(t)
			})
		})

		Convey("When the minute broadcast multiplex api fails", func() {
			mocks.Fetcher.On("FetchViewcountsForAll", ctx).Once().Return(allChannelsResp, nil)
			mocks.Spade.AssertNotCalled(t, "SendChannelConcurrents", ctx)
			mocks.Multiplex.On("StreamMinuteBroadcasts", ctx).Once().Return(nil, errors.New("mb error"))
			mocks.Statsd.On("SendGauge", mock.Anything, mock.Anything).Times(2)

			Convey("The request should not return an error", func() {
				req := &pb.UpdateSpadeViewcountReq{}

				_, err := server.UpdateSpadeViewcount(ctx, req)
				So(err, ShouldNotBeNil)
				mocks.Fetcher.AssertExpectations(t)
				mocks.Multiplex.AssertExpectations(t)
				mocks.Spade.AssertExpectations(t)
				mocks.Statsd.AssertExpectations(t)
			})
		})

		Convey("It updates spade and returns and empty response", func() {
			mocks.Fetcher.On("FetchViewcountsForAll", ctx).Once().Return(allChannelsResp, nil)
			mocks.Spade.On("SendChannelConcurrents", mock.Anything, mock.Anything, mock.Anything).Once().Return(nil)
			mocks.Multiplex.On("StreamMinuteBroadcasts", ctx).Once().Return(mbResp, nil)
			mocks.Statsd.On("SendGauge", mock.Anything, mock.Anything).Times(4)

			req := &pb.UpdateSpadeViewcountReq{}

			_, err := server.UpdateSpadeViewcount(ctx, req)
			So(err, ShouldBeNil)
			mocks.Fetcher.AssertExpectations(t)
			mocks.Multiplex.AssertExpectations(t)
			mocks.Spade.AssertExpectations(t)
			mocks.Statsd.AssertExpectations(t)
		})
	})
}

func TestUpdatePubsubViewcount(t *testing.T) {
	log.SetLevel(log.PanicLevel)
	allChannelsResp := map[string]*pb.Viewcount{
		"123456": {Count: 10},
		"222222": {Count: 6},
	}
	mbResp := []*usher.StreamMinuteBroadcast{
		{
			Channel:    "test_channel1",
			ChannelId:  int64(123456),
			ContentId:  "123456",
			CustomerId: "twitch",
			Platform:   "unknown_rtmp",
		},
		{
			Channel:    "test_channel2",
			ChannelId:  int64(222222),
			ContentId:  "222222",
			CustomerId: "twitch",
			Platform:   "unknown_rtmp",
		},
	}

	mocks := serverMocks()
	server := mockServer(mocks)
	ctx := context.Background()

	Convey("When a valid request is sent to update pubsub values", t, func() {

		Convey("When the fetcher fails", func() {
			mocks.Fetcher.On("FetchViewcountsForAll", ctx).Once().Return(nil, errors.New("api error"))
			mocks.Multiplex.AssertNotCalled(t, "StreamMinuteBroadcasts", ctx)
			mocks.Pubsub.AssertNotCalled(t, "PublishChannelConcurrents", ctx, mock.Anything, mock.Anything)
			mocks.Statsd.On("SendGauge", mock.Anything, mock.Anything).Once()

			Convey("The request should return an error", func() {
				req := &pb.UpdatePubsubViewcountReq{}

				_, err := server.UpdatePubsubViewcount(ctx, req)
				So(err, ShouldNotBeNil)
				mocks.Fetcher.AssertExpectations(t)
				mocks.Multiplex.AssertExpectations(t)
				mocks.Statsd.AssertExpectations(t)
				mocks.Pubsub.AssertExpectations(t)
			})
		})

		Convey("When the minute broadcast multiplex api fails", func() {
			mocks.Fetcher.On("FetchViewcountsForAll", ctx).Once().Return(allChannelsResp, nil)
			mocks.Multiplex.On("StreamMinuteBroadcasts", ctx).Once().Return(nil, errors.New("mb error"))
			mocks.Statsd.On("SendGauge", mock.Anything, mock.Anything).Times(2)
			mocks.Pubsub.AssertNotCalled(t, "PublishChannelConcurrents", ctx, mock.Anything, mock.Anything)

			Convey("The request should return an error", func() {
				req := &pb.UpdatePubsubViewcountReq{}

				_, err := server.UpdatePubsubViewcount(ctx, req)
				So(err, ShouldNotBeNil)
				mocks.Fetcher.AssertExpectations(t)
				mocks.Multiplex.AssertExpectations(t)
				mocks.Pubsub.AssertExpectations(t)
				mocks.Statsd.AssertExpectations(t)
			})
		})

		Convey("An empty response is returned", func() {
			mocks.Fetcher.On("FetchViewcountsForAll", ctx).Once().Return(allChannelsResp, nil)
			mocks.Multiplex.On("StreamMinuteBroadcasts", ctx).Once().Return(mbResp, nil)
			mocks.Pubsub.On("PublishChannelConcurrents", ctx, mock.Anything, mock.Anything).Once().Return(nil)
			mocks.Statsd.On("SendGauge", mock.Anything, mock.Anything).Times(3)
			req := &pb.UpdatePubsubViewcountReq{}

			_, err := server.UpdatePubsubViewcount(ctx, req)
			So(err, ShouldBeNil)
			mocks.Fetcher.AssertExpectations(t)
			mocks.Multiplex.AssertExpectations(t)
			mocks.Pubsub.AssertExpectations(t)
			mocks.Statsd.AssertExpectations(t)
		})
	})
}
