package blender

import (
	"context"
	"net/http"
	"time"

	"code.justin.tv/businessviewcount/aperture/internal/clients/stats"
	"code.justin.tv/vod/blender/rpc/blender"

	"github.com/afex/hystrix-go/hystrix"
	log "github.com/sirupsen/logrus"
)

const (
	getViewCounts = "blender.get_view_counts"
)

// Internal is an interface for retrieving watch party data
type Internal interface {
	GetViewCounts(ctx context.Context, request *blender.GetViewCountsRequest) (*blender.GetViewCountsResponse, error)
}

// Client wraps the blender protobuf client
type Client struct {
	blenderClient blender.Internal
	statsd        stats.StatSender
}

func init() {
	hystrix.Configure(map[string]hystrix.CommandConfig{
		getViewCounts: {
			Timeout:                250, // milliseconds
			MaxConcurrentRequests:  10,
			RequestVolumeThreshold: 20,
			SleepWindow:            5000, // milliseconds
			ErrorPercentThreshold:  25,
		},
	})
}

// NewClient creates a new wrapper around the blender protobuf client
func NewClient(host string, statsd stats.StatSender) *Client {
	return &Client{
		blenderClient: blender.NewInternalProtobufClient(host, &http.Client{
			Timeout: 1 * time.Second,
		}),
		statsd: statsd,
	}
}

// GetViewCounts returns view counts for all watch party channels
func (c *Client) GetViewCounts(ctx context.Context, request *blender.GetViewCountsRequest) (*blender.GetViewCountsResponse, error) {
	start := time.Now()
	var viewcounts *blender.GetViewCountsResponse
	err := hystrix.DoC(ctx, getViewCounts, func(ctx context.Context) error {
		response, err := c.blenderClient.GetViewCounts(ctx, request)
		if err != nil {
			viewcounts = &blender.GetViewCountsResponse{}
			return err
		}
		viewcounts = response
		return nil
	}, func(ctx context.Context, e error) error {
		// fallback if the request to blender returns an error or the circuit is open
		log.WithError(e).Error("hystrix circuit fallback: error getting Watch Party viewcounts from Blender")
		viewcounts = &blender.GetViewCountsResponse{}
		return nil
	})
	c.statsd.ExecutionTime(getViewCounts, time.Since(start))
	if err != nil {
		log.WithError(err).Error("Error getting Watch Party viewcounts from Blender")
	}
	return viewcounts, err
}
