package frozone

import (
	"math"
	"time"

	"code.justin.tv/businessviewcount/aperture/internal/util"
)

// CalculateViewCount calculates the viewcount for a channel that has been frozen
// Due to a picture by picture commercial session. This will return a discounted viewcount
// for a period of time, after which the viewcount will return to normal.
func CalculateViewCount(now time.Time, currentViewcount uint64, channelFreeze *util.ChannelFreeze) uint64 {
	if channelFreeze == nil {
		return currentViewcount
	}

	pbypSessionEnd := channelFreeze.CreatedAt.Add(channelFreeze.PbyPSessionLength)
	rampDownEnd := pbypSessionEnd.Add(channelFreeze.RampDownLength)
	// During ad, return viewcount at creation
	if now.After(channelFreeze.CreatedAt) && now.Before(pbypSessionEnd) {
		return channelFreeze.ViewcountAtCreation
	}

	// During ramp down period
	if now.After(pbypSessionEnd) && now.Before(rampDownEnd) {
		vcDelta := float64(currentViewcount) - float64(channelFreeze.ViewcountAtCreation)

		// If delta is within 10%, return current view count
		if math.Abs(vcDelta) < float64(channelFreeze.ViewcountAtCreation/uint64(10)) {
			return currentViewcount
		}
		// Otherwise apply modifier
		modifier := int64(rampDownEnd.Sub(now)/time.Second) / 5
		// INC-1100: if rampDownEnd.Sub(now)/time.Second < 5, modifier will == 0
		// Make sure we never divide by 0
		if modifier == 0 {
			modifier = 1
		}

		// Modifier MUST be >= 1 because we do not want to artificially inflate vcDelta
		// e.g. if modifier is 0.5, we would double vcDelta. Using an int, and making sure that
		// the modifier is not 0 will ensure that we aren't inflating vcDelta
		return channelFreeze.ViewcountAtCreation + uint64(int64(vcDelta)/modifier)
	}

	// Outside of the ramp down period, return viewcount
	return currentViewcount
}
