package frozone_test

import (
	"testing"
	"time"

	"code.justin.tv/businessviewcount/aperture/internal/clients/frozone"
	"code.justin.tv/businessviewcount/aperture/internal/util"
	. "github.com/smartystreets/goconvey/convey"
)

func TestCalculateViewCount(t *testing.T) {
	Convey("Should not change viewcount if freeze is nil", t, func() {
		ccv := uint64(100)
		vc := frozone.CalculateViewCount(time.Now().Add(time.Duration(5)*time.Second), ccv, nil)
		So(vc, ShouldEqual, ccv)
	})

	Convey("Should be able to calculate view count based on time", t, func() {
		now := time.Now()
		ccv := uint64(400)
		channelFreeze := &util.ChannelFreeze{
			CreatedAt:           now,
			ViewcountAtCreation: uint64(100),
			PbyPSessionLength:   time.Duration(60) * time.Second,
			RampDownLength:      time.Duration(30) * time.Second,
		}
		Convey("Should return viewcount at creation if in pbypSession", func() {
			inSession := now.Add(time.Duration(5) * time.Second)
			vc := frozone.CalculateViewCount(inSession, ccv, channelFreeze)
			So(vc, ShouldEqual, channelFreeze.ViewcountAtCreation)
		})

		Convey("Should return current viewcount if within 10% of vc0", func() {
			inRampDown := now.Add(time.Duration(65) * time.Second)
			vc := frozone.CalculateViewCount(inRampDown, uint64(109), channelFreeze)
			So(vc, ShouldEqual, uint64(109))
		})

		Convey("Should return a ramp down vc otherwise", func() {
			beginingRampDown := now.Add(time.Duration(61) * time.Second) // 29s remaining in ramp down
			middleRampDown := now.Add(time.Duration(75) * time.Second)   // 15s remaining in ramp down
			endRampDown := now.Add(time.Duration(85) * time.Second)      // 5s remaining in ramp down
			vc := frozone.CalculateViewCount(beginingRampDown, ccv, channelFreeze)
			So(vc, ShouldEqual, uint64(160))

			vc = frozone.CalculateViewCount(middleRampDown, ccv, channelFreeze)
			So(vc, ShouldEqual, uint64(200))

			vc = frozone.CalculateViewCount(endRampDown, ccv, channelFreeze)
			So(vc, ShouldEqual, uint64(400))
		})

		Convey("Should return a ramp down with a negative vc otherwise", func() {
			tNow := now.Add(time.Duration(75) * time.Second) // 15s remaining in ramp down
			vc := frozone.CalculateViewCount(tNow, 50, channelFreeze)
			So(vc, ShouldEqual, 84)
		})

		Convey("Should return vc if outside of ramp", func() {
			end := now.Add(time.Duration(90) * time.Second)
			vc := frozone.CalculateViewCount(end, ccv, channelFreeze)
			So(vc, ShouldEqual, uint64(400))
		})

		Convey("Should reset `modifier` to 1 if originally set to 0", func() {
			rampMinusOne := now.Add(time.Duration(89) * time.Second)   // `modifier` would equal .2 as float64, but 0 as int64
			rampMinusTwo := now.Add(time.Duration(88) * time.Second)   // `modifier` would equal .4 as float64, but 0 as int64
			rampMinusThree := now.Add(time.Duration(87) * time.Second) // `modifier` would equal .6 as float64, but 0 as int64
			rampMinusFour := now.Add(time.Duration(86) * time.Second)  // `modifier` would equal .8 as float64, but 0 as int64

			vc := frozone.CalculateViewCount(rampMinusOne, ccv, channelFreeze)
			So(vc, ShouldEqual, uint64(400))

			vc = frozone.CalculateViewCount(rampMinusTwo, ccv, channelFreeze)
			So(vc, ShouldEqual, uint64(400))

			vc = frozone.CalculateViewCount(rampMinusThree, ccv, channelFreeze)
			So(vc, ShouldEqual, uint64(400))

			vc = frozone.CalculateViewCount(rampMinusFour, ccv, channelFreeze)
			So(vc, ShouldEqual, uint64(400))
		})
	})
}
