package multiplex

import (
	"context"
	"net/http"
	"time"

	"code.justin.tv/businessviewcount/aperture/internal/clients/stats"
	"code.justin.tv/video/multiplex/rpc/multiplex"
	"code.justin.tv/video/usherapi/rpc/usher"
	log "github.com/sirupsen/logrus"
)

const (
	maxRetries        = 5
	retryDelaySeconds = 5
)

// Multiplex is an interface for sending requests to the Multiplex service
type Multiplex interface {
	StreamMinuteBroadcasts(ctx context.Context) ([]*usher.StreamMinuteBroadcast, error)
}

// Client represents an Multiplex client
type Client struct {
	multiplexClient multiplex.Multiplex
	statsd          stats.StatSender
}

// NewClient creates a new Multiplex client
func NewClient(host string, statsd stats.StatSender) *Client {
	return &Client{
		multiplexClient: multiplex.NewMultiplexProtobufClient(host, http.DefaultClient),
		statsd:          statsd,
	}
}

// StreamMinuteBroadcasts returns minute broadcast data from Multiplex. This method copies the functionality
// found in https://git.xarth.tv/video/spade_injector/tree/master/pkg/usher
func (c *Client) StreamMinuteBroadcasts(ctx context.Context) ([]*usher.StreamMinuteBroadcast, error) {
	var lastError error
	start := time.Now()
	defer func() {
		c.statsd.ExecutionTime("multiplex.stream_minute_broadcasts", time.Since(start))
	}()

	for try := 1; try <= maxRetries; try++ {
		if try != 1 {
			log.WithFields(log.Fields{
				"retry_count":    try,
				"previous_error": lastError,
			}).Warn("retrying call to multiplex after previous failure")

			time.Sleep(time.Second * retryDelaySeconds)
		}

		// Check if context is canceled or have timed out.
		if ctx.Err() != nil {
			log.WithError(ctx.Err()).Error("context done while sending request to multiplex")
			return nil, ctx.Err()
		}

		broadcasts, err := c.multiplexClient.StreamMinuteBroadcasts(ctx, &usher.StreamMinuteBroadcastsRequest{})
		if err != nil {
			log.Error("failed to send request to multiplex: ", err)
			lastError = err
			continue
		}

		return broadcasts.GetStreamMinuteBroadcasts(), nil
	}

	return nil, lastError
}
