package pubsub

import (
	"context"
	"encoding/json"
	"fmt"
	"time"

	"code.justin.tv/businessviewcount/aperture/config"
	"code.justin.tv/chat/pubsub-go-pubclient/client"
	"code.justin.tv/foundation/twitchclient"
	"golang.org/x/sync/semaphore"

	multierror "github.com/hashicorp/go-multierror"
	log "github.com/sirupsen/logrus"
)

const (
	// This limit is set by pubsub: https://git.xarth.tv/twitch/docs/blob/master/pubsub/pubsub-broker.md
	// The same limit is used by globalviewcount
	topicLimit = 1000

	channelLoginTopic = "video-playback"
	channelIDTopic    = "video-playback-by-id"

	workerPoolMaxSize = 50
)

// Publisher is an interface for publishing data to pubsub
type Publisher interface {
	PublishChannelConcurrents(ctx context.Context, env string, data map[uint64][]Recipient)
}

// Client wraps the Pubsub client.
type Client struct {
	pubsub client.PubClient
}

// NewClient instantiates a Client.
func NewClient(host string, spadeHost string) (*Client, error) {
	pubsubClient, err := client.NewPubClient(twitchclient.ClientConf{
		Host: host,
	})
	if err != nil {
		return nil, err
	}

	return &Client{
		pubsub: pubsubClient,
	}, nil
}

// PublishChannelConcurrents sends channel concurrents data to pubsub
func (c *Client) PublishChannelConcurrents(ctx context.Context, env string, data map[uint64][]Recipient) {
	channelTopicName, channelIDTopicName := c.getTopics(env)

	now := time.Now()
	timestampString := fmt.Sprintf("%f", float64(now.UnixNano())/1e9)
	workerPool := semaphore.NewWeighted(int64(workerPoolMaxSize))
	errC := make(chan error, len(data))

	for count, recipients := range data {
		pubsubUpdate := ChannelConcurrentsData{
			Type:       "viewcount",
			ServerTime: json.Number(timestampString),
			Viewers:    count,
		}

		// Pubsub expects json formatted data, but sent in a string. So we marshal
		// our data into json and then convert to a string that will be
		// sent to each topic
		msg, err := json.Marshal(pubsubUpdate)
		if err != nil {
			log.Printf("pubsub: couldn't marshal message to json: %v\n", msg)
			return
		}

		// Viewcounts are sent to a topic that contains the channel id or the channel name. We can
		// send up to the topicLimit, so we make an array of that size so we can fill it up before
		// we make a call to Publish
		topics := make([]string, 0, topicLimit)

		if err := workerPool.Acquire(ctx, 1); err != nil {
			log.WithFields(log.Fields{
				"count":      count,
				"recipients": recipients,
			}).WithError(err).Error("failed to acquire worker from worker pool")
			continue
		}

		go func(msgString string, topics []string, recipients []Recipient) {
			// We loop over each recipient and add their channel id and channel name to the appropriate
			// topic. Once we reach the topic limit, we publish that batch and flush out the slice
			for _, recipient := range recipients {
				if len(topics)+2 > topicLimit {
					err := c.pubsub.Publish(context.Background(), topics, msgString, nil)
					if err != nil {
						errC <- err
					}

					topics = make([]string, 0, topicLimit)
				}

				if recipient.ChannelID != "0" {
					topicByID := fmt.Sprintf("%s.%s", channelIDTopicName, recipient.ChannelID)
					topics = append(topics, topicByID)
				}

				if recipient.ChannelName != "" {
					topicByChannelName := fmt.Sprintf("%s.%s", channelTopicName, recipient.ChannelName)
					topics = append(topics, topicByChannelName)
				}
			}

			// Publish any leftover topics after looping
			if len(topics) > 0 {
				err := c.pubsub.Publish(context.Background(), topics, msgString, nil)
				if err != nil {
					errC <- err
				}
			}

			workerPool.Release(1)
		}(string(msg), topics, recipients)
	}

	if err := workerPool.Acquire(ctx, int64(workerPoolMaxSize)); err != nil {
		log.Error(err)
	}

	close(errC)

	var errs *multierror.Error
	for err := range errC {
		errs = multierror.Append(errs, err)
	}

	if errs.ErrorOrNil() != nil {
		log.WithError(errs).Error("failed to send some topics to pubsub")
	}
}

func (c *Client) getTopics(env string) (string, string) {
	channelTopicNameWithEnv := channelLoginTopic
	channelIDTopicNameWithEnv := channelIDTopic

	// We append the env to the event name here, so that staging events do not get sent to prod.
	// We prefer to continue sending the events on staging for testing purposes.
	if env != config.ProductionEnv {
		channelTopicNameWithEnv = fmt.Sprintf(channelLoginTopic+"-%s", env)
		channelIDTopicNameWithEnv = fmt.Sprintf(channelIDTopic+"-%s", env)
	}

	return channelTopicNameWithEnv, channelIDTopicNameWithEnv
}
