package apertureredis

import (
	"context"
	"encoding/json"
	"errors"
	"strconv"
	"time"

	"code.justin.tv/chat/rediczar"

	"code.justin.tv/businessviewcount/aperture/internal/util"
	redis "github.com/go-redis/redis/v7"
	"github.com/sirupsen/logrus"
)

// GetClient returns the redis client
func (a *client) GetClient() rediczar.ThickClient {
	return a.RedisCli
}

// SetFrozenChannel stores a freeze in redis
func (a *client) SetFrozenChannel(ctx context.Context, channelID string, channelFreeze *util.ChannelFreeze) error {
	storedFreeze, err := a.GetFrozenChannel(ctx, channelID)
	if err != nil && err != redis.Nil {
		logrus.WithError(err).Info("redis_set_frozen_channel")
		return err
	}

	// Use stored viewcount if a freeze already exists
	if storedFreeze != nil {
		channelFreeze.ViewcountAtCreation = storedFreeze.ViewcountAtCreation
	}

	if err := a.setFreeze(ctx, channelID, channelFreeze); err != nil {
		logrus.WithError(err).Info("set_frozen_channel_redis_failure")
		return err
	}

	logFreeze("set_frozen_channel_redis_success", channelID, channelFreeze)

	return nil
}

// GetFrozenChannels returns all frozen channels
func (a *client) GetFrozenChannels(ctx context.Context) (map[string]*util.ChannelFreeze, error) {
	freezes, err := a.getCleanedFreezes(ctx)
	if err != nil {
		logrus.WithError(err).Info("redis_get_frozen_channels")
		return nil, err
	}
	return freezes, nil
}

// GetFrozenChannel returns a freeze for a specific channel
func (a *client) GetFrozenChannel(ctx context.Context, channelID string) (*util.ChannelFreeze, error) {
	freezes, err := a.getCleanedFreezes(ctx)
	if err != nil {
		logrus.WithError(err).Info("redis_get_frozen_channel")
		return nil, err
	}

	freeze, ok := freezes[channelID]
	if !ok {
		return nil, redis.Nil
	}

	return freeze, nil
}

// Uses a transaction to set the freeze and it's expiration
func (a *client) setFreeze(ctx context.Context, channelID string, channelFreeze *util.ChannelFreeze) error {
	jsonBlob, err := json.Marshal(channelFreeze)
	if err != nil {
		logrus.WithError(err).Info("redis_set_frozen_channel_marshal_err")
		return err
	}

	_, err = a.RedisCli.TxPipelined(ctx, func(pipe redis.Pipeliner) error {
		// Add expiration timestamp to channel expiration set
		pipe.ZAdd(a.frozoneExpSetKey, &redis.Z{
			Score:  float64(channelFreeze.Expiration.Unix()),
			Member: channelID,
		})

		// Add freeze to channel freeze set
		pipe.HSet(a.frozoneKey, channelID, jsonBlob)
		return nil
	})

	return err
}

func (a *client) getCleanedFreezes(ctx context.Context) (map[string]*util.ChannelFreeze, error) {
	output := map[string]*util.ChannelFreeze{}

	resp, err := a.RedisCli.EvalSha(
		ctx,
		a.frozoneGetFreezesSha,
		[]string{a.frozoneExpSetKey, a.frozoneKey},
		[]string{strconv.Itoa(int(time.Now().Unix()))},
	)

	if err != nil {
		logrus.WithError(err).Info("redis_get_cleaned_freezes_error")
		return nil, err
	}

	// Redis output is formated as [key, value, key, value]
	freezeIfaces, ok := resp.([]interface{})
	if !ok {
		return nil, errors.New("Unable to parse redis output, aborting")
	}

	if len(freezeIfaces)%2 != 0 {
		return nil, errors.New("Invalid redis output")
	}

	for i := 0; i < len(freezeIfaces); i += 2 {
		channelID, ok := freezeIfaces[i].(string)
		if !ok {
			return nil, errors.New("Unable to convert channel id")
		}

		jsonString, ok := freezeIfaces[i+1].(string)
		if !ok {
			return nil, errors.New("Unable to convert redis output to string")
		}

		freeze, err := parseFreeze([]byte(jsonString))
		if err != nil {
			return nil, err
		}
		output[channelID] = freeze
	}

	logFrozenChannels(output, time.Now(), "get_frozen_channels_redis")

	return output, nil
}
