package apertureredis

import (
	"context"
	"fmt"
	"time"

	"code.justin.tv/businessviewcount/aperture/internal/clients/stats"
	"code.justin.tv/businessviewcount/aperture/internal/util"
	"code.justin.tv/chat/rediczar"
	"code.justin.tv/chat/rediczar/poolmetrics"
	"code.justin.tv/chat/rediczar/poolmetrics/statsdpoolmetrics"
	"code.justin.tv/chat/rediczar/redefault"
	"code.justin.tv/chat/rediczar/runmetrics/statsdrunmetrics"
	"github.com/sirupsen/logrus"
)

// RedisClient is an aperture specific wrapper for redis
type RedisClient interface {
	SetFrozenChannel(ctx context.Context, channlID string, channelFreeze *util.ChannelFreeze) error
	GetFrozenChannels(ctx context.Context) (map[string]*util.ChannelFreeze, error)
	GetFrozenChannel(ctx context.Context, channelID string) (*util.ChannelFreeze, error)
	GetClient() rediczar.ThickClient
}

type client struct {
	RedisCli             rediczar.ThickClient
	frozoneKey           string
	frozoneExpSetKey     string
	frozoneGetFreezesSha string
}

const (
	// AperturePrefixKey is the prefix key needed to ensure the keys are hashed to the same slot.
	AperturePrefixKey = "aperture-prefix-key"
	// FrozoneChannelsKey is the key for the channel freeze hash map.
	FrozoneChannelsKey = "frozen-channels"
	// FrozoneExpSetKey is the key for the zset that holds expiration times for freezes.
	FrozoneExpSetKey = "frozone-expired-channels"
)

// NewClient creates a new redis client for aperture
func NewClient(address string, statsCli *stats.Client, modifierKey string) (RedisClient, error) {
	clusterCli := redefault.NewClusterClient(address, nil)

	if statsCli == nil {
		redisCli := &client{
			RedisCli: &rediczar.Client{
				Commands: &rediczar.PrefixedCommands{
					Redis: clusterCli,
				},
			},
			frozoneExpSetKey: fmt.Sprintf("{%s}.%s-%s", AperturePrefixKey, FrozoneExpSetKey, modifierKey),
			frozoneKey:       fmt.Sprintf("{%s}.%s-%s", AperturePrefixKey, FrozoneChannelsKey, modifierKey),
		}

		if err := redisCli.init(); err != nil {
			return nil, err
		}
		return redisCli, nil
	}

	poolmetricsCollector := poolmetrics.Collector{
		Client: clusterCli,
		StatTracker: &statsdpoolmetrics.StatTracker{
			Stats: statsCli.NewSubStatter("redis"),
		},
	}

	if err := poolmetricsCollector.Setup(); err != nil {
		return nil, err
	}

	go func() {
		err := poolmetricsCollector.Start()
		if err != nil {
			logrus.Errorf("error starting pool metrics: %v", err)
		}
	}()

	redisCli := &client{
		RedisCli: &rediczar.Client{
			Commands: &rediczar.PrefixedCommands{
				Redis: clusterCli,
				RunMetrics: &statsdrunmetrics.StatTracker{
					Stats: statsCli.NewSubStatter("redis.commands"),
				},
			},
		},
		frozoneExpSetKey: fmt.Sprintf("{%s}.%s-%s", AperturePrefixKey, FrozoneExpSetKey, modifierKey),
		frozoneKey:       fmt.Sprintf("{%s}.%s-%s", AperturePrefixKey, FrozoneChannelsKey, modifierKey),
	}

	if err := redisCli.init(); err != nil {
		return nil, err
	}

	logrus.Info("Connection to Redis has been established")

	return redisCli, nil
}

func (a *client) init() error {
	ctx, cancel := context.WithTimeout(context.Background(), 3*time.Second)
	defer cancel()
	if _, err := a.RedisCli.Ping(ctx); err != nil {
		return err
	}

	sha, err := a.RedisCli.ScriptLoad(context.Background(), getFreezesScript)
	if err != nil {
		return err
	}

	a.frozoneGetFreezesSha = sha

	return nil
}
