package apertureredis_test

import (
	"context"
	"fmt"
	"log"
	"math/rand"
	"testing"
	"time"

	apertureredis "code.justin.tv/businessviewcount/aperture/internal/clients/redis"
	"code.justin.tv/businessviewcount/aperture/internal/util"
	redis "github.com/go-redis/redis/v7"
	. "github.com/smartystreets/goconvey/convey"
)

const redisStagingAddr = "cb-redis-aperture-staging.9jn8pu.clustercfg.usw2.cache.amazonaws.com:6379"

func TestApertureRedis(t *testing.T) {
	redisCli, err := apertureredis.NewClient(redisStagingAddr, nil, "testing")
	if err != nil {
		log.Fatal(err)
	}

	Convey("Frozone redis methods", t, func() {
		Reset(func() {
			frozoneKey := fmt.Sprintf("{%s}.%s-testing", apertureredis.AperturePrefixKey, apertureredis.FrozoneChannelsKey)
			frozoneExpSet := fmt.Sprintf("{%s}.%s-testing", apertureredis.AperturePrefixKey, apertureredis.FrozoneExpSetKey)
			_, _ = redisCli.GetClient().Del(
				context.Background(),
				frozoneKey,
			)

			_, _ = redisCli.GetClient().Del(
				context.Background(),
				frozoneExpSet,
			)
		})

		Convey("Should be able to get/set a freeze", func() {
			chanID := randomChanID()
			fakeFreeze := &util.ChannelFreeze{
				PbyPSessionLength:   time.Second * 1,
				RampDownLength:      time.Second * 2,
				CreatedAt:           time.Now(),
				Expiration:          time.Now().Add(time.Second * 4),
				ViewcountAtCreation: uint64(1234),
			}
			err := redisCli.SetFrozenChannel(context.Background(), chanID, fakeFreeze)
			So(err, ShouldBeNil)

			freeze, err := redisCli.GetFrozenChannel(context.Background(), chanID)
			So(err, ShouldBeNil)
			So(freeze.PbyPSessionLength, ShouldEqual, fakeFreeze.PbyPSessionLength)
			So(freeze.RampDownLength, ShouldEqual, fakeFreeze.RampDownLength)
			So(freeze.CreatedAt, ShouldEqual, fakeFreeze.CreatedAt)
			So(freeze.Expiration, ShouldEqual, fakeFreeze.Expiration)
			So(freeze.ViewcountAtCreation, ShouldEqual, fakeFreeze.ViewcountAtCreation)
		})

		Convey("Should not pull freeze that has expired", func() {
			chanID := randomChanID()

			freezeToExpire := &util.ChannelFreeze{
				PbyPSessionLength:   time.Second * 1,
				RampDownLength:      time.Second * 1,
				CreatedAt:           time.Now(),
				Expiration:          time.Now().Add(time.Second * 2),
				ViewcountAtCreation: uint64(1234),
			}
			err := redisCli.SetFrozenChannel(context.Background(), chanID, freezeToExpire)
			So(err, ShouldBeNil)
			time.Sleep(time.Second * 4)
			freeze, err := redisCli.GetFrozenChannel(context.Background(), chanID)
			So(err, ShouldEqual, redis.Nil)
			So(freeze, ShouldBeNil)
		})

		Convey("Should maintain original viewcount at creation if freeze is updated", func() {
			chanID := randomChanID()

			originalViewcount := uint64(100)
			originalFreeze := &util.ChannelFreeze{
				PbyPSessionLength:   time.Second * 1,
				RampDownLength:      time.Second * 1,
				CreatedAt:           time.Now(),
				Expiration:          time.Now().Add(time.Second * 5),
				ViewcountAtCreation: originalViewcount,
			}

			err := redisCli.SetFrozenChannel(context.Background(), chanID, originalFreeze)
			So(err, ShouldBeNil)

			updatingFreeze := &util.ChannelFreeze{
				PbyPSessionLength:   time.Second * 1,
				RampDownLength:      time.Second * 1,
				CreatedAt:           time.Now(),
				Expiration:          time.Now().Add(time.Second * 5),
				ViewcountAtCreation: uint64(101),
			}

			err = redisCli.SetFrozenChannel(context.Background(), chanID, updatingFreeze)
			So(err, ShouldBeNil)

			freeze, err := redisCli.GetFrozenChannel(context.Background(), chanID)
			So(err, ShouldBeNil)
			So(freeze.ViewcountAtCreation, ShouldEqual, originalViewcount)
		})

		Convey("Should be able to get all frozen channels", func() {
			chanIDA := randomChanID()
			chanIDB := randomChanID()

			chanAFreeze := &util.ChannelFreeze{
				PbyPSessionLength:   time.Second * 1,
				RampDownLength:      time.Second * 1,
				CreatedAt:           time.Now(),
				Expiration:          time.Now().Add(time.Second * 5),
				ViewcountAtCreation: uint64(12345),
			}
			chanBFreeze := &util.ChannelFreeze{
				PbyPSessionLength:   time.Second * 1,
				RampDownLength:      time.Second * 1,
				CreatedAt:           time.Now(),
				Expiration:          time.Now().Add(time.Second * 5),
				ViewcountAtCreation: uint64(123456),
			}

			err = redisCli.SetFrozenChannel(context.Background(), chanIDA, chanAFreeze)
			So(err, ShouldBeNil)
			err = redisCli.SetFrozenChannel(context.Background(), chanIDB, chanBFreeze)
			So(err, ShouldBeNil)

			freezes, err := redisCli.GetFrozenChannels(context.Background())
			So(err, ShouldBeNil)
			So(len(freezes), ShouldEqual, 2)
			So(freezes[chanIDA].ViewcountAtCreation, ShouldEqual, chanAFreeze.ViewcountAtCreation)
			So(freezes[chanIDB].ViewcountAtCreation, ShouldEqual, chanBFreeze.ViewcountAtCreation)
		})

		Convey("Should filter then remove expired freezes from hash map", func() {
			chanIDA := randomChanID()
			chanIDB := randomChanID()

			chanAFreeze := &util.ChannelFreeze{
				PbyPSessionLength:   time.Second * 1,
				RampDownLength:      time.Second * 1,
				CreatedAt:           time.Now(),
				Expiration:          time.Now().Add(time.Second * 2),
				ViewcountAtCreation: uint64(12345),
			}
			chanBFreeze := &util.ChannelFreeze{
				PbyPSessionLength:   time.Second * 1,
				RampDownLength:      time.Second * 1,
				CreatedAt:           time.Now(),
				Expiration:          time.Now().Add(time.Second * 5),
				ViewcountAtCreation: uint64(123456),
			}

			err = redisCli.SetFrozenChannel(context.Background(), chanIDA, chanAFreeze)
			So(err, ShouldBeNil)
			err = redisCli.SetFrozenChannel(context.Background(), chanIDB, chanBFreeze)
			So(err, ShouldBeNil)

			time.Sleep(time.Second * 3)
			freezes, err := redisCli.GetFrozenChannels(context.Background())
			So(err, ShouldBeNil)
			So(len(freezes), ShouldEqual, 1)
			_, ok := freezes[chanIDA]
			So(ok, ShouldBeFalse)
			_, ok = freezes[chanIDB]
			So(ok, ShouldBeTrue)

			freeze, err := redisCli.GetFrozenChannel(context.Background(), chanIDA)
			So(err, ShouldEqual, redis.Nil)
			So(freeze, ShouldBeNil)
		})
	})
}

func randomChanID() string {
	seededRand := rand.New(rand.NewSource(time.Now().UnixNano()))
	charset := "1234567890"
	b := make([]byte, 9)
	for i := range b {
		b[i] = charset[seededRand.Intn(len(charset))]
	}
	return string(b)
}
