package spade

import (
	"context"
	"fmt"

	"code.justin.tv/businessviewcount/aperture/config"
	"code.justin.tv/businessviewcount/aperture/internal/clients/stats"
	"code.justin.tv/common/spade-client-go/spade"
	"golang.org/x/sync/semaphore"

	multierror "github.com/hashicorp/go-multierror"
	log "github.com/sirupsen/logrus"
)

// Injector is an interface for sending tracking information
type Injector interface {
	SendChannelConcurrents(ctx context.Context, env string, events ViewcountEvents)
}

// Client wraps the spade client
type Client struct {
	spadeClient spade.Client
	statsClient stats.StatSender
}

const (
	workerPoolMaxSize = 2000
	spadeBatchSize    = 32
	spadeEventName    = "channel_concurrents"
	globalEventName   = "global_concurrents"
)

// NewClient creates a new wrapper around the spade client
func NewClient(spadeHost string, statsClient stats.StatSender) (*Client, error) {
	spadeClient, err := spade.NewClient()
	if err != nil {
		return nil, err
	}

	return &Client{
		spadeClient: spadeClient,
		statsClient: statsClient,
	}, nil
}

// SendChannelConcurrents writes batches of channel concurrent numbers to spade
func (c *Client) SendChannelConcurrents(ctx context.Context, env string, events ViewcountEvents) {
	go c.sendGlobalConcurrents(ctx, env, events.GlobalConcurrents)

	eventNameWithEnv := spadeEventName

	// We append the env to the event name here, so that staging events do not get written to the prod table.
	// We prefer to continue sending the events on staging for testing purposes.
	if env != config.ProductionEnv {
		eventNameWithEnv = fmt.Sprintf(spadeEventName+"-%s", env)
	}

	channels := events.ChannelConcurrents
	workerPool := semaphore.NewWeighted(int64(workerPoolMaxSize))
	errC := make(chan error, len(channels)/spadeBatchSize+1)

	for idx := 0; idx < len(channels); idx += spadeBatchSize {
		end := idx + spadeBatchSize
		if end > len(channels) {
			end = len(channels)
		}

		spadeBatch := channels[idx:end]
		spadeEvents := make([]spade.Event, len(spadeBatch))
		for idx, ccProp := range spadeBatch {
			spadeEvents[idx] = spade.Event{
				Name:       eventNameWithEnv,
				Properties: ccProp,
			}
		}

		if err := workerPool.Acquire(ctx, 1); err != nil {
			log.WithFields(log.Fields{
				"batch_start": idx,
				"batch_end":   end,
			}).WithError(err).Error("failed to acquire worker from worker pool")
			continue
		}

		go func(spadeEvents []spade.Event) {
			err := c.spadeClient.TrackEvents(ctx, spadeEvents...)
			if err != nil {
				errC <- err
			}
			workerPool.Release(1)
		}(spadeEvents)
	}

	if err := workerPool.Acquire(ctx, int64(workerPoolMaxSize)); err != nil {
		log.Error(err)
	}

	close(errC)

	var errs *multierror.Error
	for err := range errC {
		errs = multierror.Append(errs, err)
	}

	if errs.ErrorOrNil() != nil {
		log.WithError(errs).Error("failed to send some batches to spade")
	}

	c.statsClient.SendGauge(fmt.Sprintf("spade_logger.%s", eventNameWithEnv), int64(len(channels)))
}

func (c *Client) sendGlobalConcurrents(ctx context.Context, env string, globals map[string]uint64) {
	eventNameWithEnv := globalEventName
	if env != config.ProductionEnv {
		eventNameWithEnv = fmt.Sprintf(globalEventName+"-%s", env)
	}

	err := c.spadeClient.TrackEvent(ctx, eventNameWithEnv, globals)
	if err != nil {
		log.WithError(err).Error("failed to send global concurrent event to spade")
	}
}
