package stats

import (
	"fmt"
	"strconv"
	"time"

	"code.justin.tv/common/gometrics"

	metricCollector "github.com/afex/hystrix-go/hystrix/metric_collector"
	"github.com/afex/hystrix-go/plugins"
	"github.com/cactus/go-statsd-client/statsd"
	"github.com/pkg/errors"
	log "github.com/sirupsen/logrus"
)

// StatSender type alias's the exported client interface from Statsd service.
type StatSender interface {
	statsd.Statter
	ExecutionTime(statName string, duration time.Duration)
	Increment(statName string, value int64)
	SendGauge(statName string, value int64)
	SendFGauge(statName string, value float64)
}

// Client is the statsd client
type Client struct {
	statsd.Statter
}

const (
	flushInterval        = 1 * time.Second
	gometricsMonitorRate = 1 * time.Second
)

func initHystrix(host, prefix string) error {
	c, err := plugins.InitializeStatsdCollector(&plugins.StatsdCollectorConfig{
		StatsdAddr: host,
		Prefix:     prefix + ".hystrix",
	})
	if err != nil {
		return err
	}

	metricCollector.Registry.Register(c.NewStatsdCollector)
	return nil
}

// NewClient instatiates a new statsd client
func NewClient(host, env string) (*Client, error) {
	prefix := fmt.Sprintf("cb.aperture.%v", env)

	statter, err := statsd.NewBufferedClient(host, prefix, flushInterval, 512)
	if err != nil {
		return nil, errors.Wrap(err, "statsd: failed to instantiate buffered client")
	}

	log.Info(fmt.Sprintf("Connected to StatsD at %s with prefix %s", host, prefix))

	gometrics.Monitor(statter, gometricsMonitorRate)

	err = initHystrix(host, prefix)
	if err != nil {
		return nil, errors.Wrap(err, "statsd: failed to instantiate hystrix statsd collector")
	}

	return &Client{
		Statter: statter,
	}, nil
}

// ExecutionTime records the execution time given a duration. This is intended
// to be used outside of the request middleware, for query/method timing.
// This can also be called in a goroutine for performance reasons.
func (c *Client) ExecutionTime(statName string, duration time.Duration) {
	err := c.TimingDuration(statName, duration, 1.0)
	if err != nil {
		log.WithError(err).WithFields(log.Fields{
			"stat_name": statName,
			"duration":  duration,
		}).Warn("statsd: failed to send timing duration")
	}
}

// Increment increments the stat by the given value. This is intended
// to be used outside of the request middleware. This can also be called
// in a goroutine for performance reasons.
func (c *Client) Increment(statName string, value int64) {
	err := c.Inc(statName, value, 1.0)
	if err != nil {
		log.WithError(err).WithFields(log.Fields{
			"stat_name": statName,
			"value":     value,
		}).Warn("statsd: failed to send increment")
	}
}

// SendGauge sends a gauge value to statsd. This is wrapped in the client for brevity,
// because error handling is required but takes up lots of space for the caller.
func (c *Client) SendGauge(statName string, value int64) {
	err := c.Gauge(statName, value, 1.0)
	if err != nil {
		log.WithError(err).WithFields(log.Fields{
			"stat_name": statName,
			"value":     value,
		}).Warn("statsd: failed to send gauge")
	}
}

// SendFGauge sends a float64 gauge value to statsd. We use Raw here since statsd.Statter does not have
// a method supporting sending float64 gauge values. This is wrapped in the client for brevity,
// because error handling is required but takes up lots of space for the caller.
func (c *Client) SendFGauge(statName string, value float64) {
	valueStr := strconv.FormatFloat(value, 'f', -1, 64)
	err := c.Raw(statName, valueStr+"|g", 1.0)
	if err != nil {
		log.WithError(err).WithFields(log.Fields{
			"stat_name": statName,
			"value":     value,
		}).Warn("statsd: failed to send float gauge")
	}
}
