package fetcher

import (
	"context"
	"math"
	"strconv"
	"sync"
	"time"

	"code.justin.tv/businessviewcount/aperture/internal/clients/frozone"
	apertureredis "code.justin.tv/businessviewcount/aperture/internal/clients/redis"

	"code.justin.tv/businessviewcount/aperture/internal/clients/blender"
	"code.justin.tv/businessviewcount/aperture/internal/clients/memcached"
	"code.justin.tv/businessviewcount/aperture/internal/clients/stats"
	"code.justin.tv/businessviewcount/aperture/internal/clients/viewcount"
	"code.justin.tv/businessviewcount/aperture/internal/util"

	log "github.com/sirupsen/logrus"

	pb "code.justin.tv/businessviewcount/aperture/rpc/aperture"
	blenderrpc "code.justin.tv/vod/blender/rpc/blender"
)

const (
	// Max number of keys to fetch from cache
	fetchBatchSize = 100
	// Max number of concurrently executing goroutines when retrieving viewcounts
	workerPoolMaxSize = 100
	// Max time the buffers are saved in in-memory cache
	ratioBufferExpiration = 15 * time.Second
)

// Fetcher handles fetching view counts from viewcount-api and applying the cached
// channel concurrent ratios to the returned values.
type Fetcher interface {
	FetchViewcountsForAll(ctx context.Context) (map[string]*pb.Viewcount, error)
	FetchViewcountsForAllWithBlender(ctx context.Context) (map[string]*pb.Viewcount, error)
	FetchViewcountForChannel(ctx context.Context, channelID string) (*pb.Viewcount, error)
	FreezeChannel(ctx context.Context, freezeProps *pb.FreezeChannelReq) (*pb.FreezeChannelResp, error)
}

// Client implements the Fetcher interface for fetching viewcounts
type Client struct {
	BlenderClient   blender.Internal
	ViewcountClient viewcount.Viewcount
	CacheClient     memcached.Cache
	RatioBuffer     util.RatioBuffer
	// If an update is in progress, Buffers will be empty; otherwise, if will contain a true value
	RatioBufferUpdate chan bool
	StatsClient       stats.StatSender
	RedisClient       apertureredis.RedisClient
}

// Params needed to construct a Fetcher struct
type Params struct {
	Blender     blender.Internal
	Viewcount   viewcount.Viewcount
	Cache       memcached.Cache
	RatioBuffer util.RatioBuffer
	Stats       stats.StatSender
	RedisCli    apertureredis.RedisClient
}

// NewClient creates a new fetcher client
func NewClient(params *Params) Fetcher {
	client := &Client{
		BlenderClient:     params.Blender,
		ViewcountClient:   params.Viewcount,
		CacheClient:       params.Cache,
		RatioBuffer:       params.RatioBuffer,
		RatioBufferUpdate: make(chan bool, 1),
		StatsClient:       params.Stats,
		RedisClient:       params.RedisCli,
	}

	client.initBuffers()

	return client
}

// FetchViewcountForChannel retrieves the viewcount for a single channel and applies the cached ratio
func (c *Client) FetchViewcountForChannel(ctx context.Context, channelID string) (*pb.Viewcount, error) {
	channelIDInt, err := strconv.ParseUint(channelID, 10, 64)

	if err != nil {
		return nil, ErrInvalidChannelID
	}

	views, err := c.ViewcountClient.ForChannel(ctx, channelIDInt)
	if err != nil {
		return nil, err
	}

	filteredViewcountWithFreeze, err := c.applyFreezeToChannel(
		ctx,
		channelID,
		c.fetchAndApplyRatio(ctx, channelID, views.Count),
	)

	if err != nil {
		return nil, err
	}

	return &pb.Viewcount{
		Count:           filteredViewcountWithFreeze,
		CountUnfiltered: views.UnfilteredCount,
	}, nil
}

// FetchViewcountsForAll retrieves all viewcounts from the viewcount-api and then applies the secondary streaming
// ratio from the ratio cache for each channel. Returns a map of the channelID to the filtered view count
func (c *Client) FetchViewcountsForAll(ctx context.Context) (map[string]*pb.Viewcount, error) {
	views, err := c.ViewcountClient.ForAllChannels(ctx)
	if err != nil {
		return nil, err
	}
	channelIDs := getChannelIDs(views)

	// Update buffers in the background
	c.updateBuffers(channelIDs)

	ratios, _ := c.RatioBuffer.Load()

	freezes, err := c.RedisClient.GetFrozenChannels(ctx)
	if err != nil {
		return nil, err
	}

	results := make(map[string]*pb.Viewcount, len(channelIDs))

	for channelID, view := range views {
		channelIDStr := strconv.FormatUint(channelID, 10)
		if view.Count == 0 {
			results[channelIDStr] = &pb.Viewcount{CountUnfiltered: view.UnfilteredCount, Count: view.Count}
			continue
		}
		viewcount := &pb.Viewcount{CountUnfiltered: view.UnfilteredCount}
		ratio, ok := ratios[channelIDStr]
		if !ok || ratio == 1.0 {
			viewcount.Count = view.Count
		} else {
			viewcount.Count = applyRatio(ratio, view.Count)
		}

		freeze, ok := freezes[channelIDStr]
		if ok {
			// Apply frozone to viewcount
			viewcount.Count = frozone.CalculateViewCount(time.Now(), viewcount.Count, freeze)
		}

		// TODO(achou): This is temporary logging to determine what scenarios Aperture
		// ends up returning a vastly different view count from view count API.
		if float64(viewcount.Count)/float64(view.Count) < 0.05 {
			log.WithField("channel_id", channelID).WithField("og_view_count", view.Count).WithField("new_view_count", view.Count).WithField("ratio", ratio).WithField("freeze", freeze).Info("large view count discrepancy")
		}

		results[channelIDStr] = viewcount
	}

	return results, nil
}

// FetchViewcountsForAllWithBlender retrieves all viewcounts from the viewcount-api and then applies the secondary streaming
// ratio from the ratio cache for each channel. Returns a map of the channelID to the filtered view count.
// Includes Blender data, which is relevant for Spade updates only.
func (c *Client) FetchViewcountsForAllWithBlender(ctx context.Context) (map[string]*pb.Viewcount, error) {
	var wg sync.WaitGroup

	// Blender errors should not error out this function.
	var blenderCounts map[string]*pb.Viewcount
	var blenderErr error
	wg.Add(1)
	go func() {
		defer wg.Done()
		blenderCounts, blenderErr = c.fetchBlenderViewcounts(ctx)
		if blenderErr != nil {
			log.WithError(blenderErr).Error("Error getting Watch Party viewcounts from Blender")
		}
	}()

	// Fetch from viewcounts in parallel.
	var results map[string]*pb.Viewcount
	var err error
	wg.Add(1)
	go func() {
		defer wg.Done()
		results, err = c.FetchViewcountsForAll(ctx)
	}()

	// Wait for both routines to finish.
	wg.Wait()
	if err != nil {
		return nil, err
	}

	// Merge Blender results in, preferring Blender data since it's a live Watch Party.
	for channelID, blenderValue := range blenderCounts {
		results[channelID] = blenderValue
	}
	return results, nil
}

// fetchBlenderViewcounts returns viewcounts that are stored in the blender client.
func (c *Client) fetchBlenderViewcounts(ctx context.Context) (map[string]*pb.Viewcount, error) {
	resp, err := c.BlenderClient.GetViewCounts(ctx, &blenderrpc.GetViewCountsRequest{})
	if err != nil {
		return nil, err
	}

	results := make(map[string]*pb.Viewcount, len(resp.ViewCounts))
	for channelID, value := range resp.ViewCounts {
		results[channelID] = &pb.Viewcount{
			Count:           uint64(value.Value),
			CountUnfiltered: uint64(value.Value),
		}
	}
	return results, nil
}

// getRatiosFromCache retrieves the viewcount ratios from ElastiCache for all channels
func (c *Client) getRatiosFromCache(ctx context.Context, channelIDs []string) (map[string]float64, error) {
	t0 := time.Now()

	// Use a locking map so we can write the values concurrently
	innerMap := make(map[string]float64, len(channelIDs))
	lockingMap := util.LockingRatioMap{
		InnerMap: innerMap,
	}

	errs, err := util.BatchRunner(ctx, &util.BatchRunnerInput{
		FetchBatchSize:    fetchBatchSize,
		WorkerPoolMaxSize: workerPoolMaxSize,
		ListOfStr:         channelIDs,
		Runner: func(ctx context.Context, channelIDs []string) error {
			ratios, errs := c.CacheClient.GetRatioMulti(ctx, channelIDs)
			lockingMap.StoreRatios(ratios)
			// Return errs after storing ratios
			return errs
		},
	})

	if err != nil {
		c.StatsClient.Increment("fetcher.get_ratios_from_cache.error", 1)
		return nil, err
	}

	if errs.ErrorOrNil() != nil {
		log.WithError(errs).Warn("failed to get ratios from cache for some keys")
	}

	c.StatsClient.ExecutionTime("fetcher.get_ratios_from_cache.success", time.Since(t0))
	c.StatsClient.Increment("fetcher.get_ratios_from_cache.success", 1)

	return lockingMap.InnerMap, nil
}

// getRatioForChannel retrieves the viewcount ratio from the in-memory cache if it exists, and optionally fetches
// the ratio from ElastiCache if the ratios do not exist or have expired
func (c *Client) getRatioForChannel(ctx context.Context, channelID string) (float64, error) {
	ratios, timestamp := c.RatioBuffer.Load()
	if time.Now().After(timestamp.Add(ratioBufferExpiration)) {
		return c.CacheClient.GetRatio(ctx, channelID)
	}

	ratio, ok := ratios[channelID]
	if !ok {
		return 1.0, nil
	}

	return ratio, nil
}

// On creating the fetcher client, we want to populate the view count ratio buffer and the frozone buffer in memory and place the lock in the
// channel for buffer update
func (c *Client) initBuffers() {
	defer func() { c.RatioBufferUpdate <- true }()
	views, err := c.ViewcountClient.ForAllChannels(context.Background())
	if err != nil {
		log.WithError(err).Error("Error getting viewcounts when initializing ratio buffer")
		// if we failed to get from viewcounts, we initialize the buffer with a nil map
		c.RatioBuffer.Store(nil, false)
		return
	}
	chanIDs := getChannelIDs(views)
	ratios, err := c.getRatiosFromCache(context.Background(), chanIDs)
	if err != nil {
		log.WithError(err).Error("Error getting viewcount ratios when initializing ratio buffer")
		// if we failed to get from cache, we initialize the buffer with a nil map
		c.RatioBuffer.Store(nil, false)
		return
	}

	c.RatioBuffer.Store(ratios, true)
}

// updateBuffers retrieves the viewcount ratios from the in-memory buffer if it exists, and optionally fetches
// viewcount ratios from ElastiCache on a background thread if the ratios do not exist or have expired, and store it
// in the in-memory buffer
func (c *Client) updateBuffers(channelIDs []string) {
	_, lastRatioUpdate := c.RatioBuffer.Load()

	if time.Now().After(lastRatioUpdate.Add(ratioBufferExpiration)) {
		go func() {
			// attempt to acquire lock to update the ratio buffer
			select {
			case <-c.RatioBufferUpdate:
				newRatios, err := c.getRatiosFromCache(context.Background(), channelIDs)
				if err != nil {
					log.WithError(err).Warn("failed to get ratios from cache in the background")
					c.RatioBufferUpdate <- true
					return
				}

				c.RatioBuffer.Store(newRatios, true)
				c.RatioBufferUpdate <- true
			default:
				// another goroutine has the lock to update the ratio buffer
				return
			}
		}()
	}
}

// fetchAndApplyRatio retrieves the viewer ratio from cache, and multiplies the provided
// viewcount number by that ratio. We then cast the resulting float to a uint64, which is returned
func (c *Client) fetchAndApplyRatio(ctx context.Context, channelID string, views uint64) uint64 {
	if views == 0 {
		return 0
	}

	ratio, err := c.getRatioForChannel(ctx, channelID)

	// We only log a warning here if we cannot access the cache. The returned ratio is
	// always 1.0 in the case of an error, so it is still safe to apply it below
	if err != nil {
		log.WithError(err).WithField("channel_id", channelID).Warn("failed to retrieve ratio from cache")
	}

	return applyRatio(ratio, views)
}

// applyRatio applies the ratio to the view count. We round the float64 value to the nearest
// whole number, and convert to uint64
func applyRatio(ratio float64, count uint64) uint64 {
	applied := ratio * float64(count)
	return uint64(math.Round(applied))
}
