package fetcher_test

import (
	"context"
	"errors"
	"testing"
	"time"

	"code.justin.tv/businessviewcount/aperture/internal/fetcher"
	"code.justin.tv/businessviewcount/aperture/internal/mocks"
	"code.justin.tv/businessviewcount/aperture/internal/util"
	pb "code.justin.tv/businessviewcount/aperture/rpc/aperture"
	viewcount_client "code.justin.tv/video/viewcount-api/lib/client"
	"code.justin.tv/vod/blender/rpc/blender"

	log "github.com/sirupsen/logrus"
	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/mock"
)

func TestFetchViewcountForChannelWithInvalidRatioBuffer(t *testing.T) {
	log.SetLevel(log.PanicLevel)
	ctx := context.Background()
	channelID := "123456"
	channelIDInt := uint64(123456)
	singleChannelResp := &viewcount_client.Viewcount{
		Count:           10,
		UnfilteredCount: 10,
	}
	singleChannelEmptyResp := &viewcount_client.Viewcount{
		Count:           0,
		UnfilteredCount: 0,
	}

	blender := &mocks.Internal{}
	viewcount := &mocks.Viewcount{}
	viewcount.On("ForAllChannels", mock.Anything).Once().Return(nil, nil)
	cache := &mocks.Cache{}
	ratioBuffer := &mocks.RatioBuffer{}
	ratioBuffer.On("Store", mock.Anything, true).Once()
	stats := &mocks.StatSender{}
	stats.On("Increment", mock.Anything, mock.Anything).Maybe()
	stats.On("ExecutionTime", mock.Anything, mock.Anything).Maybe()

	mockRedis := &mocks.RedisClient{}
	mockRedis.On("GetFrozenChannels", mock.Anything).Return(map[string]*util.ChannelFreeze{}, nil)
	mockRedis.On("GetFrozenChannel", mock.Anything, mock.Anything).Return(&util.ChannelFreeze{}, nil)

	fetcher := fetcher.NewClient(&fetcher.Params{
		Blender:     blender,
		Viewcount:   viewcount,
		Cache:       cache,
		RatioBuffer: ratioBuffer,
		Stats:       stats,
		RedisCli:    mockRedis,
	})

	Convey("Given a valid request to fetch viewcounts for a single channel", t, func() {

		Convey("When viewcount-api fails", func() {
			ratioBuffer.On("Load").Once().Return(nil, time.Time{})
			viewcount.On("ForChannel", ctx, channelIDInt).Once().Return(nil, errors.New("viewcount error"))
			cache.AssertNotCalled(t, "GetRatio", ctx, channelID)

			Convey("It should return an error", func() {
				resp, err := fetcher.FetchViewcountForChannel(ctx, channelID)
				So(err, ShouldNotBeNil)
				So(resp, ShouldBeNil)
				cache.AssertExpectations(t)
			})
		})

		Convey("When viewcount-api succeeds and there are no viewers", func() {
			ratioBuffer.On("Load").Once().Return(nil, time.Time{})
			viewcount.On("ForChannel", ctx, channelIDInt).Once().Return(singleChannelEmptyResp, nil)
			cache.AssertNotCalled(t, "GetRatio", ctx, channelID)

			Convey("It should return 0 for the ratio without calling the cache", func() {
				resp, err := fetcher.FetchViewcountForChannel(ctx, channelID)
				So(err, ShouldBeNil)
				So(resp.Count, ShouldEqual, 0)
				So(resp.CountUnfiltered, ShouldEqual, 0)
				cache.AssertExpectations(t)
			})
		})

		Convey("When viewcount-api succeeds but the cache fails", func() {
			ratioBuffer.On("Load").Once().Return(nil, time.Time{})
			viewcount.On("ForChannel", ctx, channelIDInt).Once().Return(singleChannelResp, nil)
			cache.On("GetRatio", ctx, channelID).Once().Return(1.0, errors.New("cache error"))

			Convey("It should return 1.0 for the ratio if there are views", func() {
				resp, err := fetcher.FetchViewcountForChannel(ctx, channelID)
				So(err, ShouldBeNil)
				So(resp.Count, ShouldEqual, 10)
				So(resp.CountUnfiltered, ShouldEqual, 10)
			})
		})

		Convey("When viewcount-api succeeds and the cache succeeds", func() {
			ratioBuffer.On("Load").Once().Return(nil, time.Time{})
			viewcount.On("ForChannel", ctx, channelIDInt).Once().Return(singleChannelResp, nil)
			cache.On("GetRatio", ctx, channelID).Once().Return(float64(0.7), nil)

			Convey("It should succeed and apply the ratio", func() {
				resp, err := fetcher.FetchViewcountForChannel(ctx, channelID)
				So(err, ShouldBeNil)
				So(resp.Count, ShouldEqual, 7)
				So(resp.CountUnfiltered, ShouldEqual, 10)
			})
		})

		Convey("When viewcount-api succeeds and the cache succeeds with rounding", func() {
			ratioBuffer.On("Load").Once().Return(nil, time.Time{})
			viewcount.On("ForChannel", ctx, channelIDInt).Once().Return(singleChannelResp, nil)
			cache.On("GetRatio", ctx, channelID).Once().Return(float64(0.543432341), nil)

			Convey("It should succeed and apply the ratio", func() {
				resp, err := fetcher.FetchViewcountForChannel(ctx, channelID)
				So(err, ShouldBeNil)
				So(resp.Count, ShouldEqual, 5)
				So(resp.CountUnfiltered, ShouldEqual, 10)
			})
		})
	})
}
func TestFetchViewcountForChannelWithValidRatioBuffer(t *testing.T) {
	log.SetLevel(log.PanicLevel)
	ctx := context.Background()
	channelID := "123456"
	channelIDInt := uint64(123456)
	singleChannelResp := &viewcount_client.Viewcount{
		Count:           10,
		UnfilteredCount: 10,
	}
	allChannelsResp := map[uint64]*viewcount_client.Viewcount{
		123456: singleChannelResp,
	}
	singleChannelEmptyResp := &viewcount_client.Viewcount{
		Count:           0,
		UnfilteredCount: 0,
	}

	blender := &mocks.Internal{}
	viewcount := &mocks.Viewcount{}
	viewcount.On("ForAllChannels", mock.Anything).Once().Return(allChannelsResp, nil)
	cache := &mocks.Cache{}
	cache.On("GetRatioMulti", ctx,
		mock.MatchedBy(func(ids []string) bool {
			return assert.ElementsMatch(t, []string{channelID}, ids)
		})).Once().Return(map[string]float64{"123456": 0.7}, nil)
	ratioBuffer := &mocks.RatioBuffer{}
	ratioBuffer.On("Store", mock.Anything, true).Once()
	stats := &mocks.StatSender{}
	stats.On("Increment", mock.Anything, mock.Anything).Maybe()
	stats.On("ExecutionTime", mock.Anything, mock.Anything).Maybe()
	mockRedis := &mocks.RedisClient{}
	mockRedis.On("GetFrozenChannels", mock.Anything).Return(map[string]*util.ChannelFreeze{}, nil)
	mockRedis.On("GetFrozenChannel", mock.Anything, mock.Anything).Return(&util.ChannelFreeze{}, nil)

	fetcher := fetcher.NewClient(&fetcher.Params{
		Blender:     blender,
		Viewcount:   viewcount,
		Cache:       cache,
		RatioBuffer: ratioBuffer,
		Stats:       stats,
		RedisCli:    mockRedis,
	})

	Convey("Given a valid request to fetch viewcounts for a single channel", t, func() {

		Convey("When viewcount-api fails", func() {
			viewcount.On("ForChannel", ctx, channelIDInt).Once().Return(nil, errors.New("viewcount error"))
			ratioBuffer.AssertNotCalled(t, "Load")
			cache.AssertNotCalled(t, "GetRatio", ctx, channelID)

			Convey("It should return an error", func() {
				resp, err := fetcher.FetchViewcountForChannel(ctx, channelID)
				So(err, ShouldNotBeNil)
				So(resp, ShouldBeNil)
				cache.AssertExpectations(t)
			})

		})

		Convey("When viewcount-api succeeds and there are no viewers", func() {
			viewcount.On("ForChannel", ctx, channelIDInt).Once().Return(singleChannelEmptyResp, nil)
			ratioBuffer.AssertNotCalled(t, "Load")
			cache.AssertNotCalled(t, "GetRatio", ctx, channelID)

			Convey("It should return 0 for the ratio without calling the cache", func() {
				resp, err := fetcher.FetchViewcountForChannel(ctx, channelID)
				So(err, ShouldBeNil)
				So(resp.Count, ShouldEqual, 0)
				So(resp.CountUnfiltered, ShouldEqual, 0)
				cache.AssertExpectations(t)
			})
		})

		Convey("When viewcount-api succeeds but the cache doesn't have an entry", func() {
			viewcount.On("ForChannel", ctx, channelIDInt).Once().Return(singleChannelResp, nil)
			ratioBuffer.On("Load").Once().Return(map[string]float64{}, time.Now())

			Convey("It should return 1.0 for the ratio if there are views", func() {
				resp, err := fetcher.FetchViewcountForChannel(ctx, channelID)
				So(err, ShouldBeNil)
				So(resp.Count, ShouldEqual, 10)
				So(resp.CountUnfiltered, ShouldEqual, 10)
			})
		})

		Convey("When viewcount-api succeeds and the cache succeeds", func() {

			viewcount.On("ForChannel", ctx, channelIDInt).Once().Return(singleChannelResp, nil)
			ratioBuffer.On("Load").Once().Return(map[string]float64{
				channelID: 0.7,
			}, time.Now())

			Convey("It should succeed and apply the ratio", func() {
				resp, err := fetcher.FetchViewcountForChannel(ctx, channelID)
				So(err, ShouldBeNil)
				So(resp.Count, ShouldEqual, 7)
				So(resp.CountUnfiltered, ShouldEqual, 10)
			})
		})

		Convey("When viewcount-api succeeds and the cache succeeds with rounding", func() {
			viewcount.On("ForChannel", ctx, channelIDInt).Once().Return(singleChannelResp, nil)
			ratioBuffer.On("Load").Once().Return(map[string]float64{
				channelID: 0.543432341,
			}, time.Now())

			Convey("It should succeed and apply the ratio", func() {
				resp, err := fetcher.FetchViewcountForChannel(ctx, channelID)
				So(err, ShouldBeNil)
				So(resp.Count, ShouldEqual, 5)
				So(resp.CountUnfiltered, ShouldEqual, 10)
			})
		})

	})
}

func TestFreezeChannel(t *testing.T) {
	Convey("when FreezeChannel is called", t, func() {
		log.SetLevel(log.PanicLevel)
		ctx := context.Background()
		allChannelsResp := map[uint64]*viewcount_client.Viewcount{
			123456: {Count: 10},
			222222: {Count: 6},
			999999: {Count: 0},
		}
		blenderResp := &blender.GetViewCountsResponse{
			ViewCounts: map[string]*blender.ViewCount{
				"155875": {
					Value: 123,
				},
				"999999": {
					Value: 999,
				},
				"222222": {
					Value: 777,
				},
			},
		}

		blender := &mocks.Internal{}
		viewcount := &mocks.Viewcount{}
		cache := &mocks.Cache{}
		ratioBuffer := &mocks.RatioBuffer{}
		stats := &mocks.StatSender{}

		ratioBuffer.On("Store", mock.Anything, true).Once()
		ratioBuffer.On("Load").Return(map[string]float64{"222222": 0.5}, time.Now())

		viewcount.On("ForAllChannels", mock.Anything).Once().Return(allChannelsResp, nil)

		cache.On("GetRatioMulti", ctx,
			mock.MatchedBy(func(ids []string) bool {
				return assert.ElementsMatch(t, []string{"123456", "222222"}, ids)
			})).Once().Return(map[string]float64{"123456": 0.7}, nil)

		stats.On("Increment", mock.Anything, mock.Anything).Maybe()
		stats.On("ExecutionTime", mock.Anything, mock.Anything).Maybe()

		now := time.Now()
		myChannelFreeze := &util.ChannelFreeze{
			CreatedAt:           now,
			ViewcountAtCreation: uint64(100),
			PbyPSessionLength:   time.Duration(60) * time.Second,
			RampDownLength:      time.Duration(30) * time.Second,
		}
		mockRedis := &mocks.RedisClient{}

		fetcher := fetcher.NewClient(&fetcher.Params{
			Blender:     blender,
			Viewcount:   viewcount,
			Cache:       cache,
			RatioBuffer: ratioBuffer,
			Stats:       stats,
			RedisCli:    mockRedis,
		})

		Convey("it should use the viewcount passed in if it's valid", func() {
			mockRedis.On("SetFrozenChannel", ctx, "123456", mock.MatchedBy(func(freeze *util.ChannelFreeze) bool {
				return freeze.ViewcountAtCreation == 321
			})).Return(nil)

			resp, err := fetcher.FreezeChannel(ctx, &pb.FreezeChannelReq{
				ChannelID:    "123456",
				FreezeLength: int64(100),
				RampUpLength: int64(50),
				Viewcount:    321,
			})
			So(err, ShouldBeNil)
			So(resp, ShouldNotBeNil)
		})

		Convey("it should use the existing viewcount if a freeze is already active", func() {
			mockRedis.On("GetFrozenChannel", mock.Anything, "123456").Return(myChannelFreeze, nil)
			mockRedis.On("SetFrozenChannel", ctx, "123456", mock.MatchedBy(func(freeze *util.ChannelFreeze) bool {
				return freeze.ViewcountAtCreation == myChannelFreeze.ViewcountAtCreation
			})).Return(nil)

			resp, err := fetcher.FreezeChannel(ctx, &pb.FreezeChannelReq{
				ChannelID:    "123456",
				FreezeLength: int64(100),
				RampUpLength: int64(50),
			})
			So(err, ShouldBeNil)
			So(resp, ShouldNotBeNil)
		})

		Convey("it should get new CCV for channel if no freeze exists", func() {
			mockRedis.On("GetFrozenChannel", mock.Anything, "123456").Return(&util.ChannelFreeze{}, nil)
			mockRedis.On("SetFrozenChannel", ctx, "123456", mock.Anything).Return(nil)
			viewcount.On("ForChannel", ctx, uint64(123456)).Once().Return(&viewcount_client.Viewcount{Count: 10}, nil)

			resp, err := fetcher.FreezeChannel(ctx, &pb.FreezeChannelReq{
				ChannelID:    "123456",
				FreezeLength: int64(100),
				RampUpLength: int64(50),
			})
			So(err, ShouldBeNil)
			So(resp, ShouldNotBeNil)
		})

		Convey("it should apply the freeze to the channel - FetchViewcountForChannel", func() {
			mockRedis.On("GetFrozenChannel", mock.Anything, "123456").Return(&util.ChannelFreeze{
				CreatedAt:           now,
				ViewcountAtCreation: uint64(101),
				PbyPSessionLength:   time.Duration(60) * time.Second,
				RampDownLength:      time.Duration(30) * time.Second,
			}, nil)
			viewcount.On("ForChannel", ctx, uint64(123456)).Once().Return(&viewcount_client.Viewcount{
				Count:           100,
				UnfilteredCount: 100,
			}, nil)

			resp, err := fetcher.FetchViewcountForChannel(ctx, "123456")
			So(err, ShouldBeNil)
			So(resp, ShouldNotBeNil)
			So(resp.Count, ShouldEqual, 101)
			So(resp.CountUnfiltered, ShouldEqual, 100)
		})

		Convey("it should apply the freeze to the channel - FetchViewcountForAll", func() {
			mockRedis.On("GetFrozenChannels", mock.Anything).Return(map[string]*util.ChannelFreeze{
				"123456": {
					CreatedAt:           now,
					ViewcountAtCreation: uint64(101),
					PbyPSessionLength:   time.Duration(60) * time.Second,
					RampDownLength:      time.Duration(30) * time.Second,
				},
			}, nil)
			blender.On("GetViewCounts", mock.Anything, mock.Anything).Once().Return(blenderResp, nil)
			viewcount.On("ForAllChannels", ctx).Once().Return(allChannelsResp, nil)
			ratioBuffer.On("Load").Twice().Return(map[string]float64{"123456": 0.677642457, "222222": 0.711122343}, time.Now())

			resp, err := fetcher.FetchViewcountsForAll(ctx)
			So(err, ShouldBeNil)
			So(resp, ShouldNotBeNil)
			So(resp["123456"].Count, ShouldEqual, 101)
		})
	})
}

func TestFetchViewcountForAllWithValidCache(t *testing.T) {
	log.SetLevel(log.PanicLevel)
	ctx := context.Background()
	allChannelsResp := map[uint64]*viewcount_client.Viewcount{
		123456: {Count: 10},
		222222: {Count: 6},
	}

	blender := &mocks.Internal{}
	viewcount := &mocks.Viewcount{}
	viewcount.On("ForAllChannels", mock.Anything).Once().Return(allChannelsResp, nil)
	cache := &mocks.Cache{}
	cache.On("GetRatioMulti", ctx,
		mock.MatchedBy(func(ids []string) bool {
			return assert.ElementsMatch(t, []string{"123456", "222222"}, ids)
		})).Once().Return(map[string]float64{"123456": 0.677642457, "222222": 0.711122343}, nil)
	ratioBuffer := &mocks.RatioBuffer{}
	ratioBuffer.On("Store", mock.Anything, true).Once()
	stats := &mocks.StatSender{}
	stats.On("Increment", mock.Anything, mock.Anything).Maybe()
	stats.On("ExecutionTime", mock.Anything, mock.Anything).Maybe()
	mockRedis := &mocks.RedisClient{}
	mockRedis.On("GetFrozenChannels", mock.Anything).Return(map[string]*util.ChannelFreeze{}, nil)
	mockRedis.On("GetFrozenChannel", mock.Anything, mock.Anything).Return(&util.ChannelFreeze{}, nil)

	fetcher := fetcher.NewClient(&fetcher.Params{
		Blender:     blender,
		Viewcount:   viewcount,
		Cache:       cache,
		RatioBuffer: ratioBuffer,
		Stats:       stats,
		RedisCli:    mockRedis,
	})

	Convey("Given a valid request to fetch viewcounts for all channels", t, func() {

		Convey("When viewcount-api fails", func() {
			viewcount.On("ForAllChannels", ctx).Once().Return(nil, errors.New("viewcount error"))
			ratioBuffer.AssertNotCalled(t, "Load")

			Convey("It should return an error", func() {
				resp, err := fetcher.FetchViewcountsForAll(ctx)
				So(err, ShouldNotBeNil)
				So(resp, ShouldBeNil)
				cache.AssertExpectations(t)
			})
		})

		Convey("When the viewcount succeeds and the cache has a value", func() {
			viewcount.On("ForAllChannels", ctx).Once().Return(allChannelsResp, nil)
			ratioBuffer.On("Load").Twice().Return(map[string]float64{"123456": 0.8, "222222": 0.5}, time.Now())

			Convey("It should succeed and apply the ratio to all channels", func() {
				resp, err := fetcher.FetchViewcountsForAll(ctx)
				So(err, ShouldBeNil)
				So(resp["123456"].Count, ShouldEqual, 8)
				So(resp["222222"].Count, ShouldEqual, 3)
			})
		})

		Convey("When the viewcount succeeds and the cache has a value with rounding", func() {
			viewcount.On("ForAllChannels", ctx).Once().Return(allChannelsResp, nil)
			ratioBuffer.On("Load").Twice().Return(map[string]float64{"123456": 0.677642457, "222222": 0.711122343}, time.Now())

			Convey("It should succeed and apply the ratio to all channels", func() {
				resp, err := fetcher.FetchViewcountsForAll(ctx)
				So(err, ShouldBeNil)
				So(resp["123456"].Count, ShouldEqual, 7)
				So(resp["222222"].Count, ShouldEqual, 4)
			})
		})
	})
}

func TestFetchViewcountsForAllBlenderWithValidCache(t *testing.T) {
	log.SetLevel(log.PanicLevel)
	ctx := context.Background()
	allChannelsResp := map[uint64]*viewcount_client.Viewcount{
		123456: {Count: 10},
		222222: {Count: 6},
		999999: {Count: 33},
	}
	blenderResp := &blender.GetViewCountsResponse{
		ViewCounts: map[string]*blender.ViewCount{
			"155875": {
				Value: 123,
			},
			"999999": {
				Value: 999,
			},
		},
	}

	blender := &mocks.Internal{}
	viewcount := &mocks.Viewcount{}
	viewcount.On("ForAllChannels", mock.Anything).Once().Return(allChannelsResp, nil)
	cache := &mocks.Cache{}
	cache.On("GetRatioMulti", ctx,
		mock.MatchedBy(func(ids []string) bool {
			return assert.ElementsMatch(t, []string{"123456", "222222", "999999"}, ids)
		})).Once().Return(map[string]float64{"123456": 0.677642457, "222222": 0.711122343}, nil)
	ratioBuffer := &mocks.RatioBuffer{}
	ratioBuffer.On("Store", mock.Anything, true).Once()
	stats := &mocks.StatSender{}
	stats.On("Increment", mock.Anything, mock.Anything).Maybe()
	stats.On("ExecutionTime", mock.Anything, mock.Anything).Maybe()
	mockRedis := &mocks.RedisClient{}
	mockRedis.On("GetFrozenChannels", mock.Anything).Return(map[string]*util.ChannelFreeze{}, nil)

	fetcher := fetcher.NewClient(&fetcher.Params{
		Blender:     blender,
		Viewcount:   viewcount,
		Cache:       cache,
		RatioBuffer: ratioBuffer,
		Stats:       stats,
		RedisCli:    mockRedis,
	})

	Convey("Given a valid request to fetch viewcounts for all channels", t, func() {

		Convey("When viewcount-api fails", func() {
			blender.On("GetViewCounts", mock.Anything, mock.Anything).Once().Return(blenderResp, nil)
			viewcount.On("ForAllChannels", ctx).Once().Return(nil, errors.New("viewcount error"))
			ratioBuffer.AssertNotCalled(t, "Load")

			Convey("It should return an error", func() {
				resp, err := fetcher.FetchViewcountsForAllWithBlender(ctx)
				So(err, ShouldNotBeNil)
				So(resp, ShouldBeNil)
				cache.AssertExpectations(t)
			})
		})

		Convey("When the viewcount succeeds and the cache has a value", func() {
			blender.On("GetViewCounts", mock.Anything, mock.Anything).Once().Return(blenderResp, nil)
			viewcount.On("ForAllChannels", ctx).Once().Return(allChannelsResp, nil)
			ratioBuffer.On("Load").Twice().Return(map[string]float64{"123456": 0.8, "222222": 0.5}, time.Now())

			Convey("It should succeed and apply the ratio to all channels", func() {
				resp, err := fetcher.FetchViewcountsForAllWithBlender(ctx)
				So(err, ShouldBeNil)
				So(resp["123456"].Count, ShouldEqual, 8)
				So(resp["222222"].Count, ShouldEqual, 3)
				So(resp["999999"].Count, ShouldEqual, 999)
				So(resp["155875"].Count, ShouldEqual, 123)
			})
		})

		Convey("When the viewcount succeeds and the cache has a value with rounding", func() {
			blender.On("GetViewCounts", mock.Anything, mock.Anything).Once().Return(blenderResp, nil)
			viewcount.On("ForAllChannels", ctx).Once().Return(allChannelsResp, nil)
			ratioBuffer.On("Load").Twice().Return(map[string]float64{"123456": 0.677642457, "222222": 0.711122343}, time.Now())

			Convey("It should succeed and apply the ratio to all channels", func() {
				resp, err := fetcher.FetchViewcountsForAllWithBlender(ctx)
				So(err, ShouldBeNil)
				So(resp["123456"].Count, ShouldEqual, 7)
				So(resp["222222"].Count, ShouldEqual, 4)
				So(resp["999999"].Count, ShouldEqual, 999)
				So(resp["155875"].Count, ShouldEqual, 123)
			})
		})

		Convey("When blender fails", func() {
			blender.On("GetViewCounts", mock.Anything, mock.Anything).Once().Return(nil, errors.New("blender error"))
			viewcount.On("ForAllChannels", ctx).Once().Return(allChannelsResp, nil)
			ratioBuffer.On("Load").Twice().Return(map[string]float64{"123456": 0.8, "222222": 0.5}, time.Now())
			resp, err := fetcher.FetchViewcountsForAll(ctx)
			So(err, ShouldBeNil)
			So(resp["123456"].Count, ShouldEqual, 8)
			So(resp["222222"].Count, ShouldEqual, 3)
			So(resp["155875"], ShouldBeNil)
		})
	})
}
