package fetcher

import (
	"context"
	"strconv"
	"time"

	redis "github.com/go-redis/redis/v7"
	log "github.com/sirupsen/logrus"
	"github.com/twitchtv/twirp"

	"code.justin.tv/businessviewcount/aperture/internal/util"
	pb "code.justin.tv/businessviewcount/aperture/rpc/aperture"
)

// FreezeChannel freezes the viewcount for a given channel for a specified period of time
func (c *Client) FreezeChannel(ctx context.Context, freezeProps *pb.FreezeChannelReq) (*pb.FreezeChannelResp, error) {
	t0 := time.Now()

	// Use the viewcount input to avoid calling the viewcount API. The viewcount input is expected to have the ratio
	// applied already.
	viewcountWithRatio := freezeProps.Viewcount
	if viewcountWithRatio == 0 {
		log.WithField("channel_id", freezeProps.ChannelID).Info("received 0 viewcount")
		count, err := c.getViewcountFromFrozone(ctx, freezeProps.ChannelID)
		if err != nil {
			return nil, twirp.InternalError(err.Error())
		}

		viewcountWithRatio = c.fetchAndApplyRatio(
			ctx,
			freezeProps.ChannelID,
			count,
		)
	}

	pbypLengthInSeconds := time.Second * time.Duration(freezeProps.FreezeLength)
	rampDownLengthInSeconds := time.Second * time.Duration(freezeProps.RampUpLength)
	freeze := &util.ChannelFreeze{
		ViewcountAtCreation: viewcountWithRatio,
		CreatedAt:           t0,
		RampDownLength:      rampDownLengthInSeconds,
		PbyPSessionLength:   pbypLengthInSeconds,
		Expiration:          t0.Add(rampDownLengthInSeconds + pbypLengthInSeconds),
	}

	err := c.RedisClient.SetFrozenChannel(ctx, freezeProps.ChannelID, freeze)
	if err != nil {
		return nil, twirp.InternalError(err.Error())
	}

	return &pb.FreezeChannelResp{}, nil
}

// https://jira.twitch.com/browse/ADVAX-1024
// Check to see if a freeze exists, and use the stored viewcount. Otherwise fetch the latest viewcount
func (c *Client) getViewcountFromFrozone(ctx context.Context, channelID string) (uint64, error) {
	chanIDUint, err := strconv.ParseUint(channelID, 10, 64)
	if err != nil {
		return 0, twirp.NewError(twirp.Malformed, err.Error())
	}

	freeze, err := c.RedisClient.GetFrozenChannel(ctx, channelID)
	if err != nil && err != redis.Nil {
		return 0, err
	}

	if freeze != nil {
		return freeze.ViewcountAtCreation, nil
	}

	view, err := c.ViewcountClient.ForChannel(ctx, chanIDUint)
	if err != nil {
		log.WithError(err).Error("freeze_channel_viewcount_err")
		return 0, twirp.InternalError(err.Error())
	}

	return view.Count, nil
}
