package util

import (
	"context"
	"errors"

	"golang.org/x/sync/semaphore"

	log "github.com/sirupsen/logrus"

	multierror "github.com/hashicorp/go-multierror"
)

// BatchRunnerInput is the struct that constructs a batch runnner
type BatchRunnerInput struct {
	FetchBatchSize    int
	WorkerPoolMaxSize int64
	ListOfStr         []string
	Runner            func(context.Context, []string) error
}

// BatchRunner runs operations over a list of strings
// Returns a list of errors from goroutines and an
// error if we are unable to acquire a worker from the worker pool
func BatchRunner(ctx context.Context, opts *BatchRunnerInput) (*multierror.Error, error) {
	workerPool := semaphore.NewWeighted(opts.WorkerPoolMaxSize)
	var errs *multierror.Error
	errC := make(chan error, len(opts.ListOfStr)/opts.FetchBatchSize+1)

	for i := 0; i < len(opts.ListOfStr); i += opts.FetchBatchSize {
		batchStart := i
		batchEnd := i + opts.FetchBatchSize
		if batchEnd > len(opts.ListOfStr) {
			batchEnd = len(opts.ListOfStr)
		}
		if err := workerPool.Acquire(ctx, 1); err != nil {
			log.WithFields(log.Fields{
				"batch_start": batchStart,
				"batch_end":   batchEnd,
			}).WithError(err).Error("failed to acquire worker from worker pool")
			errC <- err
			continue
		}

		go func(strs []string) {
			err := opts.Runner(ctx, strs)
			if err != nil {
				errC <- err
			}
			workerPool.Release(1)
		}(opts.ListOfStr[batchStart:batchEnd])
	}

	if err := workerPool.Acquire(ctx, opts.WorkerPoolMaxSize); err != nil {
		return nil, errors.New("cannot acquire worker from pool")
	}
	close(errC)
	for err := range errC {
		errs = multierror.Append(errs, err)
	}
	if errs.ErrorOrNil() != nil {
		return errs, nil
	}
	return nil, nil
}
