package util

import (
	"sync"
)

// LockingRatioMap allows use to write to a map concurrently by locking it first and unlocking it when we finish.
// There are four main reasons to use this over sync.Map:
//
// 1. We lose all type safety when using interface{} as the type for keys and values.
// 2. We can range over the InnerMap easier after the keys have been written.
// 3. sync.Map may or may not have worse performance than a simple lock. At least this struct is predictable.
// 4. From the golang docs: "The Map type is specialized. Most code should use a plain Go map instead,
//    with separate locking or coordination, for better type safety and to make it easier to maintain other invariants along with the map content."
type LockingRatioMap struct {
	sync.Mutex
	InnerMap map[string]float64
}

// StoreRatios stores a map of ratios in the locking map, given strings as the map keys.
func (l *LockingRatioMap) StoreRatios(ratios map[string]float64) {
	l.Lock()
	for key, value := range ratios {
		l.InnerMap[key] = value
	}
	l.Unlock()
}
