package util

import (
	"errors"
	"fmt"
	"strconv"
	"strings"

	"code.justin.tv/businessviewcount/aperture/internal/clients/pubsub"
	pb "code.justin.tv/businessviewcount/aperture/rpc/aperture"
	"code.justin.tv/video/usherapi/rpc/usher"
	multierror "github.com/hashicorp/go-multierror"
	log "github.com/sirupsen/logrus"
)

// ConvertViewcountsToPubsub converts viewcount and usher data to a map that can be used by pubsub.
// This attempts to keep functional parity with the globalviewcount service publishing logic
// (see https://git.xarth.tv/video/globalviewcount/blob/master/publish/pubsub/pubsub.go).
//
// The returned value is a mapping from viewcount to recipient array. We key on the viewcount size in order
// to save the number of requests made to pubsub, according to globalviewcount.
func ConvertViewcountsToPubsub(viewcounts map[string]*pb.Viewcount, mbData []*usher.StreamMinuteBroadcast) map[uint64][]pubsub.Recipient {
	viewcountMap := make(map[uint64][]pubsub.Recipient)
	var errs *multierror.Error

	for _, mbEvent := range mbData {
		channelName := mbEvent.GetChannel()
		if channelName == "" {
			errs = multierror.Append(errs, errors.New("missing channel name in mb data"))
			continue
		}
		channelID := mbEvent.GetChannelId()
		if channelID == 0 {
			if strings.HasPrefix(channelName, "lvs.") {
				continue
			}
			errs = multierror.Append(errs, fmt.Errorf("missing channel id in mb data: %s", channelName))
			continue
		}

		// Check if this is a primary job, only send pubsub if it is primary
		if mbEvent.GetJobType() != "primary" {
			continue
		}

		channelIDStr := strconv.FormatInt(channelID, 10)
		viewCount, viewCountExists := viewcounts[channelIDStr]
		if !viewCountExists {
			continue
		}

		recipient := pubsub.Recipient{
			ChannelID:   channelIDStr,
			ChannelName: channelName,
		}

		currentCount := viewCount.Count

		currentRecipients, ok := viewcountMap[currentCount]
		if !ok {
			viewcountMap[currentCount] = []pubsub.Recipient{recipient}
		} else {
			viewcountMap[currentCount] = append(currentRecipients, recipient)
		}
	}

	if errs.ErrorOrNil() != nil {
		log.WithError(errs).Warn("failed to convert some data to pubsub map")
	}

	return viewcountMap
}
