package util_test

import (
	"testing"

	"code.justin.tv/businessviewcount/aperture/internal/util"

	pb "code.justin.tv/businessviewcount/aperture/rpc/aperture"
	"code.justin.tv/video/usherapi/rpc/usher"
	log "github.com/sirupsen/logrus"
	. "github.com/smartystreets/goconvey/convey"
)

func TestConvertViewcountsToPubsub(t *testing.T) {
	log.SetLevel(log.PanicLevel)

	testChannelID1 := int64(123456)
	testChannelID1Str := "123456"
	testChannelName1 := "test_channel1"
	testChannel1Viewcount := uint64(10)

	testChannelID2 := int64(222222)
	testChannelID2Str := "222222"
	testChannelName2 := "test_channel2"
	testChannel2Viewcount := uint64(6)

	allChannelsData := map[string]*pb.Viewcount{
		"123456": {Count: testChannel1Viewcount},
		"222222": {Count: testChannel2Viewcount},
	}

	Convey("When given viewcount and minute broadcast data", t, func() {

		Convey("It skips minute broadcast data when the channel doesn't exist", func() {
			mbData := []*usher.StreamMinuteBroadcast{
				{
					Channel:   testChannelName1,
					ChannelId: testChannelID1,
					JobType:   "primary",
				},
				// no channel in this entry, wont be published to pubsub
				{
					ChannelId: testChannelID2,
					JobType:   "primary",
				},
			}

			pubsubData := util.ConvertViewcountsToPubsub(allChannelsData, mbData)
			So(len(pubsubData), ShouldEqual, 1)
			So(len(pubsubData[testChannel1Viewcount]), ShouldEqual, 1)
			So(pubsubData[testChannel2Viewcount], ShouldBeNil)
		})

		Convey("It skips minute broadcast data when the channel id doesn't exist", func() {
			mbData := []*usher.StreamMinuteBroadcast{
				{
					Channel:   testChannelName1,
					ChannelId: testChannelID1,
					JobType:   "primary",
				},
				// no channel id in this entry, wont be published to pubsub
				{
					Channel: testChannelName2,
					JobType: "primary",
				},
			}

			pubsubData := util.ConvertViewcountsToPubsub(allChannelsData, mbData)
			So(len(pubsubData), ShouldEqual, 1)
			So(len(pubsubData[testChannel1Viewcount]), ShouldEqual, 1)
			So(pubsubData[testChannel2Viewcount], ShouldBeNil)
		})

		Convey("It skips minute broadcast data when the channel's transcode job type is not primary", func() {
			mbData := []*usher.StreamMinuteBroadcast{
				{
					Channel:   testChannelName1,
					ChannelId: testChannelID1,
					JobType:   "primary",
				},
				// wrong job type in this entry, wont be published to pubsub
				{
					Channel:   testChannelName2,
					ChannelId: testChannelID2,
					JobType:   "secondary",
				},
			}

			pubsubData := util.ConvertViewcountsToPubsub(allChannelsData, mbData)
			So(len(pubsubData), ShouldEqual, 1)
			So(len(pubsubData[testChannel1Viewcount]), ShouldEqual, 1)
			So(pubsubData[testChannel2Viewcount], ShouldBeNil)
		})

		Convey("It skips minute broadcast data when the channel viewcounts don't exist", func() {
			mbData := []*usher.StreamMinuteBroadcast{
				{
					Channel:   testChannelName1,
					ChannelId: testChannelID1,
					JobType:   "primary",
				},
				// channel id doesn't exist in allChannelsData
				{
					Channel:   testChannelName2,
					ChannelId: int64(999999),
					JobType:   "primary",
				},
			}

			pubsubData := util.ConvertViewcountsToPubsub(allChannelsData, mbData)
			So(len(pubsubData), ShouldEqual, 1)
			So(len(pubsubData[testChannel1Viewcount]), ShouldEqual, 1)
			So(pubsubData[testChannel2Viewcount], ShouldBeNil)
		})

		Convey("It converts them to a pubsub expected map with correct data", func() {
			mbData := []*usher.StreamMinuteBroadcast{
				{
					Channel:   testChannelName1,
					ChannelId: testChannelID1,
					JobType:   "primary",
				},
				{
					Channel:   testChannelName2,
					ChannelId: testChannelID2,
					JobType:   "primary",
				},
			}

			pubsubData := util.ConvertViewcountsToPubsub(allChannelsData, mbData)
			So(len(pubsubData), ShouldEqual, 2)
			So(len(pubsubData[testChannel1Viewcount]), ShouldEqual, 1)
			So(pubsubData[testChannel1Viewcount][0].ChannelID, ShouldEqual, testChannelID1Str)
			So(pubsubData[testChannel1Viewcount][0].ChannelName, ShouldEqual, testChannelName1)

			So(len(pubsubData[testChannel2Viewcount]), ShouldEqual, 1)
			So(pubsubData[testChannel2Viewcount][0].ChannelID, ShouldEqual, testChannelID2Str)
			So(pubsubData[testChannel2Viewcount][0].ChannelName, ShouldEqual, testChannelName2)
		})

		Convey("It converts them to a pubsub expected map with correct data when there are duplicate viewcounts", func() {
			testChannelName3 := "test_channel3"
			testChannelID3 := int64(333333)
			testChannelID3Str := "333333"

			// Two test ids have the same viewcount, so they should appear in the same map entry
			allChannelsData = map[string]*pb.Viewcount{
				"123456": {Count: testChannel1Viewcount},
				"333333": {Count: testChannel1Viewcount},
				"222222": {Count: testChannel2Viewcount},
			}

			mbData := []*usher.StreamMinuteBroadcast{
				{
					Channel:   testChannelName1,
					ChannelId: testChannelID1,
					JobType:   "primary",
				},
				{
					Channel:   testChannelName2,
					ChannelId: testChannelID2,
					JobType:   "primary",
				},
				{
					Channel:   testChannelName3,
					ChannelId: testChannelID3,
					JobType:   "primary",
				},
			}

			pubsubData := util.ConvertViewcountsToPubsub(allChannelsData, mbData)
			So(len(pubsubData), ShouldEqual, 2)
			So(len(pubsubData[testChannel1Viewcount]), ShouldEqual, 2)
			So(pubsubData[testChannel1Viewcount][0].ChannelID, ShouldEqual, testChannelID1Str)
			So(pubsubData[testChannel1Viewcount][0].ChannelName, ShouldEqual, testChannelName1)
			So(pubsubData[testChannel1Viewcount][1].ChannelID, ShouldEqual, testChannelID3Str)
			So(pubsubData[testChannel1Viewcount][1].ChannelName, ShouldEqual, testChannelName3)

			So(len(pubsubData[testChannel2Viewcount]), ShouldEqual, 1)
			So(pubsubData[testChannel2Viewcount][0].ChannelID, ShouldEqual, testChannelID2Str)
			So(pubsubData[testChannel2Viewcount][0].ChannelName, ShouldEqual, testChannelName2)
		})
	})
}
