package util

import (
	"sync/atomic"
	"time"
)

// RatioBuffer is a simple in-memory singleton buffer that stores ratios as a map and records the storage time
type RatioBuffer interface {
	Store(m map[string]float64, valid bool)
	Load() (map[string]float64, time.Time)
}

// ratioBuffer is a wrapper around atomic.Value, which stores the ratio map and the storage time as a pointer to
// bufferItem.
// Note that atomic.Value requires the values stored be consistently typed
type ratioBuffer struct {
	item atomic.Value
}

type bufferItem struct {
	Ratios    map[string]float64
	Timestamp time.Time
}

// NewRatioBuffer makes a new RatioBuffer
func NewRatioBuffer() RatioBuffer {
	return &ratioBuffer{}
}

// Store stores a ratio map to the ratio buffer. The map stored must not be modified after Store is called.
func (b *ratioBuffer) Store(m map[string]float64, valid bool) {
	item := bufferItem{Ratios: m, Timestamp: time.Now()}
	if !valid {
		// if the given map is invalid (e.g. a dummy map not fetched from cache), we set the timestamp to nil so we will
		// attempt fetching again on next call
		item.Timestamp = time.Time{}
	}

	b.item.Store(item)
}

// Load loads the ratio map and when it was stored from buffer. The map loaded is read-only.
func (b *ratioBuffer) Load() (map[string]float64, time.Time) {
	result, ok := b.item.Load().(bufferItem)
	if !ok {
		// we either never stored an item in the buffer or we stored the wrong type.
		return nil, time.Time{}
	}
	return result.Ratios, result.Timestamp
}
