package util

import (
	"errors"
	"fmt"
	"strconv"
	"strings"
	"time"

	"code.justin.tv/businessviewcount/aperture/internal/clients/spade"

	pb "code.justin.tv/businessviewcount/aperture/rpc/aperture"
	"code.justin.tv/video/usherapi/rpc/usher"
	multierror "github.com/hashicorp/go-multierror"
	log "github.com/sirupsen/logrus"
)

const globalTotalKey = "total"

// ConvertViewcountsToSpade converts a map of channelIDs -> viewcounts to an array of properties
// matching the fields that we expect to send to spade. mbData contains structured data that
// comes from an multiplex endpoint (see the Multiplex client for details). This additions contains channel information
// that is logged to spade along with view counts.
//
// The processing of mbData is meant to match what already exists in spade_injector, to maintain parity. This
// applies to the types of fields, and which fields we can tolerate as missing data without skipping the entire entry.
func ConvertViewcountsToSpade(viewcounts map[string]*pb.Viewcount, mbData []*usher.StreamMinuteBroadcast) spade.ViewcountEvents {
	allData := filterSpadeData(mbData)
	channelProps := make([]spade.ChannelConcurrentProperties, 0, len(allData))
	globalProps := map[string]uint64{
		globalTotalKey: 0,
	}

	timestamp := time.Now().Unix()

	for _, mbEvent := range allData {
		ccProp := spade.ChannelConcurrentProperties{}
		ccProp.Time = timestamp

		channelName := mbEvent.GetChannel()
		ccProp.Channel = channelName

		channelID := mbEvent.GetChannelId()

		ccProp.ChannelID = channelID

		channelIDStr := strconv.FormatInt(channelID, 10)
		viewCount, viewCountExists := viewcounts[channelIDStr]
		if !viewCountExists {
			viewCount = &pb.Viewcount{
				Count:           0,
				CountUnfiltered: 0,
			}
		}
		ccProp.Total = viewCount.Count
		ccProp.EdgeManifestCount = viewCount.Count

		// ContentID and customerID are allowed be blank if data is missing
		ccProp.ContentID = mbEvent.GetContentId()

		platform := mbEvent.GetPlatform()
		if platform == "spectre" {
			ccProp.ContentMode = "playlist"
		} else if platform == "prime_video" {
			ccProp.ContentMode = "prime_video_watch_party"
		} else {
			ccProp.ContentMode = "live"
		}

		ccProp.CustomerID = mbEvent.GetCustomerId()
		// Watch parties are not counted towards global concurrents.
		if ccProp.CustomerID == "twitch" && platform != "prime_video" {
			globalProps[globalTotalKey] = globalProps[globalTotalKey] + viewCount.Count
		}

		ccProp.DistinctID = strconv.Itoa(int(timestamp))

		channelProps = append(channelProps, ccProp)
	}

	return spade.ViewcountEvents{
		ChannelConcurrents: channelProps,
		GlobalConcurrents:  globalProps,
	}
}

// Filters out datapoints which are missing fields needed.
func filterSpadeData(mbData []*usher.StreamMinuteBroadcast) []*usher.StreamMinuteBroadcast {
	var errs *multierror.Error
	allData := make([]*usher.StreamMinuteBroadcast, 0, len(mbData))

	skippedChannelCount := 0
	for _, mbEvent := range mbData {
		channelName := mbEvent.GetChannel()
		if channelName == "" {
			errs = multierror.Append(errs, errors.New("missing channel name in mb data"))
			skippedChannelCount++
			continue
		}

		channelID := mbEvent.GetChannelId()
		if channelID == 0 {
			if strings.HasPrefix(channelName, "lvs.") {
				continue
			}
			errs = multierror.Append(errs, fmt.Errorf("missing channel id in mb data: %s", channelName))
			skippedChannelCount++
			continue
		}

		// Check if this is a primary job, only send spade event if it is primary
		if mbEvent.GetJobType() != "primary" {
			log.WithField("channel_id", mbEvent.GetChannelId()).
				WithField("job_type", mbEvent.GetJobType()).
				Info("skipped spade event due to transcode job type not primary")
			continue
		}

		allData = append(allData, mbEvent)
	}

	if errs.ErrorOrNil() != nil {
		log.WithError(errs).
			WithField("skipped_channels", skippedChannelCount).
			Warn("failed to convert some data to spade properties")
	}
	return allData
}
