package util_test

import (
	"testing"

	"code.justin.tv/businessviewcount/aperture/internal/util"

	pb "code.justin.tv/businessviewcount/aperture/rpc/aperture"
	"code.justin.tv/video/usherapi/rpc/usher"
	log "github.com/sirupsen/logrus"
	. "github.com/smartystreets/goconvey/convey"
)

func TestConvertViewcountsToSpade(t *testing.T) {
	log.SetLevel(log.PanicLevel)
	allChannelsData := map[string]*pb.Viewcount{
		"123456": {Count: 10},
		"222222": {Count: 6},
	}

	testChannelID1 := int64(123456)
	testChannelID1Str := "123456"
	testChannelName1 := "test_channel1"
	testChannelID2 := int64(222222)
	testChannelID2Str := "222222"
	testChannelName2 := "test_channel2"
	testCustomerID := "twitch"

	Convey("When given viewcount and minute broadcast data", t, func() {

		Convey("It skips minute broadcast data when the channel doesn't exist", func() {
			mbData := []*usher.StreamMinuteBroadcast{
				{
					Channel:    testChannelName1,
					ChannelId:  testChannelID1,
					ContentId:  testChannelID1Str,
					CustomerId: testCustomerID,
					JobType:    "primary",
					Platform:   "unknown_rtmp",
				},
				// no channel in this entry, wont be sent to spade
				{
					ChannelId:  testChannelID2,
					ContentId:  testChannelID2Str,
					CustomerId: testCustomerID,
					JobType:    "primary",
					Platform:   "spectre",
				},
			}

			spadeData := util.ConvertViewcountsToSpade(allChannelsData, mbData)
			So(len(spadeData.ChannelConcurrents), ShouldEqual, 1)
			So(spadeData.GlobalConcurrents["total"], ShouldEqual, 10)
		})

		Convey("It skips minute broadcast data when the channel id doesn't exist", func() {
			mbData := []*usher.StreamMinuteBroadcast{
				{
					Channel:    testChannelName1,
					ChannelId:  testChannelID1,
					ContentId:  testChannelID1Str,
					CustomerId: testCustomerID,
					JobType:    "primary",
					Platform:   "unknown_rtmp",
				},
				// no channel id in this entry, wont be sent to spade
				{
					Channel:    testChannelName2,
					ContentId:  testChannelID2Str,
					CustomerId: testCustomerID,
					JobType:    "primary",
					Platform:   "spectre",
				},
			}

			spadeData := util.ConvertViewcountsToSpade(allChannelsData, mbData)
			So(len(spadeData.ChannelConcurrents), ShouldEqual, 1)
			So(spadeData.GlobalConcurrents["total"], ShouldEqual, 10)
		})

		Convey("It skips minute broadcast data when the channel's transcode job type is not primary", func() {
			mbData := []*usher.StreamMinuteBroadcast{
				{
					Channel:    testChannelName1,
					ChannelId:  testChannelID1,
					ContentId:  testChannelID1Str,
					CustomerId: testCustomerID,
					JobType:    "primary",
					Platform:   "unknown_rtmp",
				},
				// wrong job type in this entry, wont be sent to spade
				{
					Channel:    testChannelName2,
					ChannelId:  testChannelID2,
					ContentId:  testChannelID2Str,
					CustomerId: testCustomerID,
					JobType:    "secondary",
					Platform:   "spectre",
				},
			}

			spadeData := util.ConvertViewcountsToSpade(allChannelsData, mbData)
			So(len(spadeData.ChannelConcurrents), ShouldEqual, 1)
			So(spadeData.GlobalConcurrents["total"], ShouldEqual, 10)
		})

		Convey("It does not skip minute broadcast data when the channel viewcounts don't exist", func() {
			mbData := []*usher.StreamMinuteBroadcast{
				{
					Channel:    testChannelName1,
					ChannelId:  testChannelID1,
					ContentId:  testChannelID1Str,
					CustomerId: testCustomerID,
					JobType:    "primary",
					Platform:   "unknown_rtmp",
				},
				// channel id doesn't exist in allChannelsData
				{
					Channel:    testChannelName2,
					ChannelId:  999999,
					ContentId:  "999999",
					CustomerId: testCustomerID,
					JobType:    "primary",
					Platform:   "spectre",
				},
			}

			spadeData := util.ConvertViewcountsToSpade(allChannelsData, mbData)
			So(len(spadeData.ChannelConcurrents), ShouldEqual, 2)
			So(spadeData.ChannelConcurrents[1].EdgeManifestCount, ShouldEqual, 0)
			So(spadeData.ChannelConcurrents[1].Total, ShouldEqual, 0)
			So(spadeData.GlobalConcurrents["total"], ShouldEqual, 10)
		})

		Convey("It leaves content id blank if it is missing in minute broadcast", func() {
			mbData := []*usher.StreamMinuteBroadcast{
				{
					Channel:    testChannelName1,
					ChannelId:  testChannelID1,
					CustomerId: testCustomerID,
					JobType:    "primary",
					Platform:   "unknown_rtmp",
				},
			}

			spadeData := util.ConvertViewcountsToSpade(allChannelsData, mbData)
			So(len(spadeData.ChannelConcurrents), ShouldEqual, 1)
			So(spadeData.ChannelConcurrents[0].ContentID, ShouldEqual, "")
			So(spadeData.GlobalConcurrents["total"], ShouldEqual, 10)
		})

		Convey("It leaves customer id blank if it is missing in minute broadcast", func() {
			mbData := []*usher.StreamMinuteBroadcast{
				{
					Channel:   testChannelName1,
					ChannelId: testChannelID1,
					ContentId: testChannelID1Str,
					JobType:   "primary",
					Platform:  "unknown_rtmp",
				},
			}

			spadeData := util.ConvertViewcountsToSpade(allChannelsData, mbData)
			So(len(spadeData.ChannelConcurrents), ShouldEqual, 1)
			So(spadeData.ChannelConcurrents[0].CustomerID, ShouldEqual, "")

			// No global concurrent data because we skip customer ids that aren't "twitch"
			So(spadeData.GlobalConcurrents["total"], ShouldEqual, 0)
		})

		Convey("It converts them to spade properties with correct data", func() {
			mbData := []*usher.StreamMinuteBroadcast{
				{
					Channel:    testChannelName1,
					ChannelId:  testChannelID1,
					ContentId:  testChannelID1Str,
					CustomerId: testCustomerID,
					JobType:    "primary",
					Platform:   "unknown_rtmp",
				},
				{
					Channel:    testChannelName2,
					ChannelId:  testChannelID2,
					ContentId:  testChannelID2Str,
					CustomerId: testCustomerID,
					JobType:    "primary",
					Platform:   "spectre",
				},
			}

			spadeData := util.ConvertViewcountsToSpade(allChannelsData, mbData)
			So(len(spadeData.ChannelConcurrents), ShouldEqual, 2)

			So(spadeData.ChannelConcurrents[0], ShouldNotBeNil)
			So(spadeData.ChannelConcurrents[0].Channel, ShouldEqual, testChannelName1)
			So(spadeData.ChannelConcurrents[0].ChannelID, ShouldEqual, int64(123456))
			So(spadeData.ChannelConcurrents[0].ContentID, ShouldEqual, testChannelID1Str)
			So(spadeData.ChannelConcurrents[0].ContentMode, ShouldEqual, "live")
			So(spadeData.ChannelConcurrents[0].CustomerID, ShouldEqual, testCustomerID)
			So(spadeData.ChannelConcurrents[0].EdgeManifestCount, ShouldEqual, uint64(10))
			So(spadeData.ChannelConcurrents[0].Total, ShouldEqual, uint64(10))
			So(spadeData.ChannelConcurrents[0].Time, ShouldNotBeNil)
			So(spadeData.ChannelConcurrents[0].DistinctID, ShouldNotBeNil)

			So(spadeData.ChannelConcurrents[1], ShouldNotBeNil)
			So(spadeData.ChannelConcurrents[1].Channel, ShouldEqual, testChannelName2)
			So(spadeData.ChannelConcurrents[1].ChannelID, ShouldEqual, int64(222222))
			So(spadeData.ChannelConcurrents[1].ContentID, ShouldEqual, testChannelID2Str)
			So(spadeData.ChannelConcurrents[1].ContentMode, ShouldEqual, "playlist")
			So(spadeData.ChannelConcurrents[1].CustomerID, ShouldEqual, testCustomerID)
			So(spadeData.ChannelConcurrents[1].EdgeManifestCount, ShouldEqual, uint64(6))
			So(spadeData.ChannelConcurrents[1].Total, ShouldEqual, uint64(6))
			So(spadeData.ChannelConcurrents[1].Time, ShouldNotBeNil)
			So(spadeData.ChannelConcurrents[1].DistinctID, ShouldNotBeNil)

			So(spadeData.GlobalConcurrents["total"], ShouldEqual, 16)
		})
	})
}
