package main

import (
	"context"
	"os"

	"github.com/aws/aws-lambda-go/lambda"

	"code.justin.tv/businessviewcount/aperture/config"
	"code.justin.tv/businessviewcount/aperture/internal/clients/secrets"
	"code.justin.tv/businessviewcount/aperture/internal/clients/stats"
	ka "code.justin.tv/businessviewcount/aperture/internal/kinesisanalytics"

	log "github.com/sirupsen/logrus"
)

// errorStreamLoggerLambda implements the lambda to write errors from error_stream in kinesis analytics app to rollbar
type errorStreamLoggerLambda struct {
	statsdClient stats.StatSender
}

// Handler writes errors to rollbar
func (l *errorStreamLoggerLambda) Handler(ctx context.Context, event ka.Event) (ka.Response, error) {
	log.WithFields(log.Fields{
		"InvocationID": event.InvocationID,
		"RecordCount":  len(event.Records),
	}).Info("received kinesis analytics event from error_stream")

	var response ka.Response

	errs := make([]string, len(event.Records))
	for i, record := range event.Records {
		errs[i] = string(record.Data)
		response.Records = append(response.Records, ka.ResponseRecord{
			RecordID: record.RecordID,
			Result:   ka.DeliveryStateOk,
		})
	}

	log.WithField("errors", errs).Error("received kinesis analytics error from error_stream")

	return response, nil
}

func main() {
	env := os.Getenv("ENVIRONMENT")
	if env == "" {
		log.Fatal("error_stream_logger: no environment found in env")
		return
	}

	conf := &config.Config{
		Environment: env,
	}

	err := conf.Load()
	if err != nil {
		log.Fatal("error_stream_logger: could not load config: ", err)
		return
	}

	secretManager, err := secrets.NewManager()
	if err != nil {
		log.Fatal("error_stream_logger: could not create secrets manager: ", err)
		return
	}

	statsdClient, err := stats.NewClient(conf.StatsdHost.Get(), env)
	if err != nil {
		log.Fatal("error_stream_logger: could not create statsd client: ", err)
		return
	}

	config.SetupRollbarLogging(secretManager,
		conf.RollbarTokenSecretName.Get(),
		conf.RollbarTokenSecretKey.Get(),
		env)

	errorStreamLogger := errorStreamLoggerLambda{
		statsdClient,
	}

	lambda.Start(errorStreamLogger.Handler)
}
