package main

import (
	"context"
	"encoding/json"
	"errors"
	"os"
	"strconv"
	"strings"
	"time"

	"code.justin.tv/businessviewcount/aperture/config"
	"code.justin.tv/businessviewcount/aperture/internal/clients/memcached"
	"code.justin.tv/businessviewcount/aperture/internal/clients/secrets"
	"code.justin.tv/businessviewcount/aperture/internal/clients/stats"
	"code.justin.tv/businessviewcount/aperture/internal/kinesisanalytics"

	"github.com/aws/aws-lambda-go/lambda"
	log "github.com/sirupsen/logrus"
	"golang.org/x/sync/semaphore"
)

// Timestamp type for time in Kinesis Analytics data.
// Kinesis Analytics outputs time as "2006-01-02 15:04:05.999", which Golang cannot automatically parse as time.Time.
type Timestamp struct {
	Time time.Time
}

const (
	sqlTimestampFormat = "2006-01-02 15:04:05.999"
	// a record is retried if it gets a ProcessingFailed response; we only want to retry 5 times per record
	maxRecordRetries  = 5
	workerPoolMaxSize = 10
)

// UnmarshalJSON parses a given byte array as a Timestamp.
func (t *Timestamp) UnmarshalJSON(b []byte) error {
	var err error
	// the raw bytes has surrounding quotation marks, e.g. "2006-01-02 15:04:05.999"
	s := strings.Trim(string(b), "\"")
	if s == "null" {
		t.Time = time.Time{}
		log.Warning("unmarshal warning: timestamp is null")
		return nil
	}

	t.Time, err = time.Parse(sqlTimestampFormat, s)
	return err
}

// MarshalJSON converts a Timestamp to a string according to the SQL Timestamp Format.
// nolint:unparam
func (t Timestamp) MarshalJSON() ([]byte, error) {
	return []byte(strconv.Quote(t.Time.Format(sqlTimestampFormat))), nil
}

type channelRatioData struct {
	ChannelID     string    `json:"channel_id"`
	Ratio         float64   `json:"ratio"`
	Total         int       `json:"total"`
	FilteredCount int       `json:"filtered_count"`
	Minutes       int       `json:"minutes"`
	WindowEndTime Timestamp `json:"window_end_time"`
}

// Lambda implements the lambda to output minute-watched ratio to elasticache
type Lambda struct {
	Cache  memcached.Cache
	Config *config.Config
	Statsd stats.StatSender
}

func (l *Lambda) shouldLog(channelID string) bool {
	whitelist := strings.Split(l.Config.LoggingChannelsWhitelist.Get(), ",")
	for _, channel := range whitelist {
		if channelID == channel {
			return true
		}
	}
	return false
}

func (l *Lambda) logData(data channelRatioData) {
	if !l.shouldLog(data.ChannelID) {
		return
	}
	statName := "output_mw_ratio_to_elasticache_lambda.ratio." + data.ChannelID
	l.Statsd.SendFGauge(statName, data.Ratio)
}

func (l *Lambda) processChannelRatioData(ctx context.Context, data channelRatioData) (string, error) {
	ratio := data.Ratio
	if ratio < 0 || ratio > 1 {
		return kinesisanalytics.DeliveryStateOk, errors.New("value error: ratio must be between 0 and 1")
	}
	err := l.Cache.SetRatio(ctx, data.ChannelID, ratio)
	if err != nil {
		return kinesisanalytics.DeliveryStateProcessingFailed, err
	}
	l.logData(data)
	return kinesisanalytics.DeliveryStateOk, nil
}

// Handler updates the squad stream secondary player ratio for each channel in Elasticache
func (l *Lambda) Handler(ctx context.Context, event kinesisanalytics.Event) (kinesisanalytics.Response, error) {
	log.WithFields(log.Fields{
		"InvocationID": event.InvocationID,
		"RecordCount":  len(event.Records),
	}).Info("received kinesis analytics event for ratio")

	var response kinesisanalytics.Response

	workerPool := semaphore.NewWeighted(int64(workerPoolMaxSize))
	responseRecordC := make(chan kinesisanalytics.ResponseRecord, len(event.Records))

	for _, record := range event.Records {
		if err := workerPool.Acquire(ctx, 1); err != nil {
			log.WithFields(log.Fields{
				"recordID": record.RecordID,
			}).WithError(err).Error("failed to acquire worker from worker pool")
			responseRecord := kinesisanalytics.ResponseRecord{
				RecordID: record.RecordID,
				Result:   kinesisanalytics.DeliveryStateProcessingFailed,
			}
			responseRecordC <- responseRecord
			continue
		}

		go func(record kinesisanalytics.EventRecord) {
			responseRecord := kinesisanalytics.ResponseRecord{
				RecordID: record.RecordID,
			}

			var data channelRatioData
			err := json.Unmarshal(record.Data, &data)
			if err != nil {
				log.WithError(err).Errorf("unmarshal error: failed to unmarshal record %v", record.RecordID)
				responseRecord.Result = kinesisanalytics.DeliveryStateProcessingFailed
			} else {
				responseRecord.Result, err = l.processChannelRatioData(context.Background(), data)
				if err != nil {
					log.WithFields(log.Fields{
						"recordID":  record.RecordID,
						"channelID": data.ChannelID,
					}).WithError(err).Error("process error: failed to process record")
				}
			}

			// if the record is retried for the 5th time, mark it as successful
			if record.LambdaDeliveryRecordMetadata.RetryHint >= maxRecordRetries {
				responseRecord.Result = kinesisanalytics.DeliveryStateOk
			}
			responseRecordC <- responseRecord
			workerPool.Release(1)
		}(record)
	}
	// We acquire all workers here to ensure that this function does not
	// return until all goroutines have finished executing
	if err := workerPool.Acquire(ctx, int64(workerPoolMaxSize)); err != nil {
		log.Error(err)
	}
	close(responseRecordC)

	for record := range responseRecordC {
		response.Records = append(response.Records, record)
	}

	return response, nil
}

func main() {
	env := os.Getenv("ENVIRONMENT")
	if env == "" {
		log.Fatal("kinesis_lambda: no environment found in env")
		return
	}

	conf := &config.Config{
		Environment: env,
	}

	err := conf.Load()
	if err != nil {
		log.Fatal("kinesis_lambda: could not load config: ", err)
		return
	}

	secretManager, err := secrets.NewManager()
	if err != nil {
		log.Fatal("kinesis_lambda: could not create secrets manager: ", err)
		return
	}

	config.SetupRollbarLogging(secretManager,
		conf.RollbarTokenSecretName.Get(),
		conf.RollbarTokenSecretKey.Get(),
		env)

	cache := memcached.NewCache(
		conf.CacheAddress.Get(),
		int(conf.CacheMaxIdleConns.Get()),
		conf.CachePollInterval.Get(),
		conf.CacheTimeout.Get(),
		conf.CacheExpirationDuration,
	)

	statsdClient, err := stats.NewClient(conf.StatsdHost.Get(), env)
	if err != nil {
		log.Fatal("kinesis_lambda: could not create statsd client: ", err)
		return
	}

	l := &Lambda{
		Cache:  cache,
		Config: conf,
		Statsd: statsdClient,
	}

	lambda.Start(l.Handler)
}
