package main

import (
	"context"
	"encoding/json"
	"errors"
	"testing"
	"time"

	"code.justin.tv/businessviewcount/aperture/config"
	"code.justin.tv/businessviewcount/aperture/internal/kinesisanalytics"
	"code.justin.tv/businessviewcount/aperture/internal/mocks"

	"code.justin.tv/hygienic/distconf"
	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"

	log "github.com/sirupsen/logrus"
)

type mockLambdaParams struct {
	Cache  *mocks.Cache
	Config *config.Config
	Statsd *mocks.StatSender
}

func lambdaMocks(conf *distconf.Distconf) mockLambdaParams {
	cache := &mocks.Cache{}
	config := &config.Config{
		LoggingChannelsWhitelist: conf.Str("logging.whitelist", ""),
	}
	statsd := &mocks.StatSender{}

	return mockLambdaParams{
		Cache:  cache,
		Config: config,
		Statsd: statsd,
	}
}

func mockLambda(params mockLambdaParams) *Lambda {
	return &Lambda{
		Cache:  params.Cache,
		Config: params.Config,
		Statsd: params.Statsd,
	}
}

func TestHandler(t *testing.T) {
	log.SetLevel(log.PanicLevel)
	channelID := "123456"
	ratio := float64(0.56)
	total := 100
	filteredCount := 56
	minutes := 5
	invocationID := "9876"
	applicationARN := "arn"
	recordID := "1"
	timestamp := Timestamp{Time: time.Now()}

	localConf := &distconf.InMemory{}
	conf := &distconf.Distconf{Readers: []distconf.Reader{localConf}}
	mocks := lambdaMocks(conf)
	lambda := mockLambda(mocks)
	ctx := context.Background()

	Convey("Given a valid Kinesis Analytics Event to output ratios to ElastiCache", t, func() {

		Convey("When the data is malformed", func() {
			record := kinesisanalytics.EventRecord{
				RecordID: recordID,
				Data:     []byte("hi"),
			}
			event := kinesisanalytics.Event{
				InvocationID:   invocationID,
				ApplicationARN: applicationARN,
				Records:        []kinesisanalytics.EventRecord{record},
			}

			Convey("The handler should mark the record as processing failed", func() {
				response, err := lambda.Handler(ctx, event)
				So(err, ShouldBeNil)
				So(response, ShouldNotBeNil)
				So(response.Records, ShouldNotBeEmpty)
				So(response.Records[0].RecordID, ShouldEqual, recordID)
				So(response.Records[0].Result, ShouldEqual, kinesisanalytics.DeliveryStateProcessingFailed)
			})

			Convey("When the record has been retried max times", func() {
				event.Records[0].LambdaDeliveryRecordMetadata = kinesisanalytics.LambdaDeliveryRecordMetadata{RetryHint: maxRecordRetries}

				Convey("The handler should mark the record as delivered", func() {
					response, err := lambda.Handler(ctx, event)
					So(err, ShouldBeNil)
					So(response, ShouldNotBeNil)
					So(response.Records, ShouldNotBeEmpty)
					So(response.Records[0].RecordID, ShouldEqual, recordID)
					So(response.Records[0].Result, ShouldEqual, kinesisanalytics.DeliveryStateOk)
				})
			})
		})

		Convey("When the ratio is out of bounds", func() {
			data := channelRatioData{
				ChannelID:     channelID,
				Ratio:         5.1,
				Total:         total,
				FilteredCount: filteredCount,
				Minutes:       minutes,
				WindowEndTime: timestamp,
			}

			dataStr, err := json.Marshal(data)
			So(err, ShouldBeNil)

			record := kinesisanalytics.EventRecord{
				RecordID: recordID,
				Data:     dataStr,
			}
			event := kinesisanalytics.Event{
				InvocationID:   invocationID,
				ApplicationARN: applicationARN,
				Records:        []kinesisanalytics.EventRecord{record},
			}

			Convey("The handler should mark the record as delivered", func() {
				response, err := lambda.Handler(ctx, event)
				So(err, ShouldBeNil)
				So(response, ShouldNotBeNil)
				So(response.Records, ShouldNotBeEmpty)
				So(response.Records[0].RecordID, ShouldEqual, recordID)
				So(response.Records[0].Result, ShouldEqual, kinesisanalytics.DeliveryStateOk)
			})
		})

		Convey("When the cache fails", func() {
			mocks.Cache.On("SetRatio", ctx, channelID, ratio).Once().Return(errors.New("api error"))

			data := channelRatioData{
				ChannelID:     channelID,
				Ratio:         ratio,
				Total:         total,
				FilteredCount: filteredCount,
				Minutes:       minutes,
				WindowEndTime: timestamp,
			}

			dataStr, err := json.Marshal(data)
			So(err, ShouldBeNil)

			record := kinesisanalytics.EventRecord{
				RecordID: recordID,
				Data:     dataStr,
			}
			event := kinesisanalytics.Event{
				InvocationID:   invocationID,
				ApplicationARN: applicationARN,
				Records:        []kinesisanalytics.EventRecord{record},
			}

			Convey("The handler should mark the record as processing failed", func() {
				response, err := lambda.Handler(ctx, event)
				So(err, ShouldBeNil)
				So(response, ShouldNotBeNil)
				So(response.Records, ShouldNotBeEmpty)
				So(response.Records[0].RecordID, ShouldEqual, recordID)
				So(response.Records[0].Result, ShouldEqual, kinesisanalytics.DeliveryStateProcessingFailed)
			})

			Convey("When the record has been retried max times", func() {
				event.Records[0].LambdaDeliveryRecordMetadata = kinesisanalytics.LambdaDeliveryRecordMetadata{RetryHint: maxRecordRetries}

				Convey("The handler should mark the record as delivered", func() {
					response, err := lambda.Handler(ctx, event)
					So(err, ShouldBeNil)
					So(response, ShouldNotBeNil)
					So(response.Records, ShouldNotBeEmpty)
					So(response.Records[0].RecordID, ShouldEqual, recordID)
					So(response.Records[0].Result, ShouldEqual, kinesisanalytics.DeliveryStateOk)
				})
			})
		})

		Convey("When the record is valid", func() {
			mocks.Cache.On("SetRatio", ctx, channelID, ratio).Once().Return(nil)

			data := channelRatioData{
				ChannelID:     channelID,
				Ratio:         ratio,
				Total:         total,
				FilteredCount: filteredCount,
				Minutes:       minutes,
				WindowEndTime: timestamp,
			}

			dataStr, err := json.Marshal(data)
			So(err, ShouldBeNil)

			record := kinesisanalytics.EventRecord{
				RecordID: recordID,
				Data:     dataStr,
			}
			event := kinesisanalytics.Event{
				InvocationID:   invocationID,
				ApplicationARN: applicationARN,
				Records:        []kinesisanalytics.EventRecord{record},
			}

			Convey("The handler should not return an error and mark the record as delivered", func() {
				response, err := lambda.Handler(ctx, event)
				So(err, ShouldBeNil)
				So(response, ShouldNotBeNil)
				So(response.Records, ShouldNotBeEmpty)
				So(response.Records[0].RecordID, ShouldEqual, recordID)
				So(response.Records[0].Result, ShouldEqual, kinesisanalytics.DeliveryStateOk)
				mocks.Cache.AssertExpectations(t)
			})
		})

		Convey("When record is valid and the channel should be logged", func() {
			channelID := "234567"

			// add channel to the logging whitelist
			err := localConf.Write("logging.whitelist", []byte(channelID))
			So(err, ShouldBeNil)

			mocks.Cache.On("SetRatio", ctx, channelID, ratio).Once().Return(nil)
			mocks.Statsd.On("SendFGauge", mock.Anything, ratio).Once().Return(nil)

			data := channelRatioData{
				ChannelID:     channelID,
				Ratio:         ratio,
				Total:         total,
				FilteredCount: filteredCount,
				Minutes:       minutes,
				WindowEndTime: timestamp,
			}

			dataStr, err := json.Marshal(data)
			So(err, ShouldBeNil)

			record := kinesisanalytics.EventRecord{
				RecordID: recordID,
				Data:     dataStr,
			}
			event := kinesisanalytics.Event{
				InvocationID:   invocationID,
				ApplicationARN: applicationARN,
				Records:        []kinesisanalytics.EventRecord{record},
			}

			Convey("The handler should not return an error and mark the record as delivered", func() {
				response, err := lambda.Handler(ctx, event)
				So(err, ShouldBeNil)
				So(response, ShouldNotBeNil)
				So(response.Records, ShouldNotBeEmpty)
				So(response.Records[0].RecordID, ShouldEqual, recordID)
				So(response.Records[0].Result, ShouldEqual, kinesisanalytics.DeliveryStateOk)
				mocks.Cache.AssertExpectations(t)
				mocks.Statsd.AssertExpectations(t)
			})
		})
	})
}
