package main

import (
	"context"
	"os"
	"time"

	"github.com/aws/aws-lambda-go/lambda"
	"golang.org/x/sync/errgroup"

	"code.justin.tv/businessviewcount/aperture/config"
	"code.justin.tv/businessviewcount/aperture/internal/clients/secrets"
	"code.justin.tv/businessviewcount/aperture/internal/clients/stats"
	"code.justin.tv/foundation/twitchclient"

	pb "code.justin.tv/businessviewcount/aperture/rpc/aperture"
	log "github.com/sirupsen/logrus"
)

// pubsubSenderLambda implements the lambda to call the pubsub publisher and send channel concurrents to pubsub
type pubsubSenderLambda struct {
	apertureClient pb.Aperture
	statsdClient   stats.StatSender
}

// Handler to publish channel concurrents to pubsub. The handler is called every
// minute, set by a cloudwatch event.
// This function actually calls UpdatePubsubViewcount twice, sleeping 30s in between. This
// behavior is essentially a hack to get around the fact that cloudwatch rules cannot be executed
// any faster than every 1 minute, but we need pubsub to be updated every 30s.
//
// Alternative solutions to this problem are not sufficient to support this service:
// 1. Use beanstalk worker envs:
//    Workers have proven unreliable in the past. sqsd can periodically have networking issues
//    which can bring down the env without any changes. Further, leader election in multi-instance
//    envs has failed in the past and causing duplicate or no events to be handled. This is unacceptable
//    for this service.
// 2. Use an EC2 instance with a cron schedule to call lambda-invoke
//    This is very non standard and would require additional infrastructure and documentation. It's worse
//    for maintenance and also limits access to the cron schedule to those who can ssh into the instance.
//    Additional infrastructure and points of failure cause more overhead for onboarding, moving the project,
//    and handling errors + alarms.
// 3. Multiple lambdas:
//    Similar to above, extra infra chained together in a non-standard way is not a clean solution, and adds in
//    multiple points of failure that need to have alarms and logging and extensive documentation to
//    determine how to manage it. If we're going to hack around the cloudwatch limit, we might as well do
//    it in the way that results in the simplest management.
func (l *pubsubSenderLambda) Handler(ctx context.Context) error {
	req := &pb.UpdatePubsubViewcountReq{}
	var group errgroup.Group

	group.Go(func() error {
		return l.update(ctx, req)
	})

	time.Sleep(30 * time.Second)

	group.Go(func() error {
		return l.update(ctx, req)
	})

	return group.Wait()
}

func (l *pubsubSenderLambda) update(ctx context.Context, req *pb.UpdatePubsubViewcountReq) error {
	start := time.Now()
	_, err := l.apertureClient.UpdatePubsubViewcount(ctx, req)
	elapsed := time.Since(start)
	l.statsdClient.ExecutionTime("pubsub_sender.execution_time", elapsed)

	return err
}

func main() {
	env := os.Getenv("ENVIRONMENT")
	if env == "" {
		log.Fatal("pubsub_sender: no environment found in env")
		return
	}

	conf := &config.Config{
		Environment: env,
	}

	err := conf.Load()
	if err != nil {
		log.Fatal("pubsub_sender: could not load config: ", err)
		return
	}

	secretManager, err := secrets.NewManager()
	if err != nil {
		log.Fatal("pubsub_sender: could not create secrets manager: ", err)
		return
	}

	statsdClient, err := stats.NewClient(conf.StatsdHost.Get(), env)
	if err != nil {
		log.Fatal("pubsub_sender: could not create statsd client: ", err)
		return
	}

	config.SetupRollbarLogging(secretManager,
		conf.RollbarTokenSecretName.Get(),
		conf.RollbarTokenSecretKey.Get(),
		env)

	apertureHost := conf.ApertureHost.Get()
	clientConf := twitchclient.ClientConf{
		Transport: twitchclient.TransportConf{
			MaxIdleConnsPerHost: 100,
		},
		Host: apertureHost,
	}

	apertureClient := pb.NewApertureProtobufClient(apertureHost, twitchclient.NewHTTPClient(clientConf))
	pubsubSender := pubsubSenderLambda{
		apertureClient,
		statsdClient,
	}

	lambda.Start(pubsubSender.Handler)
}
